-- mySeq.e
-- v1.0
-- 2010-11-13
-- sequence functions

-- function leftTrim(sequence s)
--   removes space chars from the beginning of a string
-- function rightTrim(sequence s)
--   removes space chars from the end of a string
-- function trim(sequence s)
--   removes space chars from both ends of a string
-- function rightPad(sequence s, atom len)
--   returns a string right-padded to length len
-- function leftPad(sequence s, atom len)
--   returns a string left-padded to length len
-- function right(sequence s, integer n)
--   returns the n characters at the right of string s

include myTypes.e
include myConv.e
include myDebug.e

------------------------------------------------------------------------------

global function leftTrim(sequence s)
-- remove space chars from the beginning of a string
  integer l

  l = length(s)
  if l=0 then return "" end if
  for i=1 to l do
    if s[i] != 32 then
      return s[i..l]
    end if
  end for
  return ""
end function

------------------------------------------------------------------------------

global function rightTrim(sequence s)
-- remove space chars from the end of a string
  integer l

  l = length(s)
  if l=0 then return "" end if
  for i=l to 1 by -1 do
    if s[i] != 32 then
      return s[1..i]
    end if
  end for
  return ""
end function

------------------------------------------------------------------------------

global function trim(sequence s)
-- remove space chars from both ends of a string
  return leftTrim(rightTrim(s))
end function

------------------------------------------------------------------------------

global function rightPad(sequence s, atom len)
--Return a string right-padded to length len

  if length(s) >= len then
    return s
  else
    return s & repeat(' ', len-length(s))
  end if
end function

------------------------------------------------------------------------------

global function leftPad(sequence s, atom len)
--Return a string left-padded to length len

  if length(s) >= len then
    return s
  else
    return repeat(' ', len-length(s)) & s
  end if
end function

------------------------------------------------------------------------------

global function right(sequence s, integer n)
-- returns the n characters at the right of string s
  integer l

  l = length(s)
  if (l=0) or (n=0) then return "" end if
  if n >= l then return s end if
  return s[l-n+1..l]
end function

------------------------------------------------------------------------------

global function maxIndex(sequence lst)
-- returns index of the highest value from the list lst
-- if unsuccessful, returns 0
  atom maxi

  for i = 1 to length(lst) do
    if sequence(lst[i]) then
      return 0
    elsif i=1 then
      maxi = i
    elsif lst[i] > lst[maxi] then
      maxi = i
    end if
  end for
  return maxi
end function

------------------------------------------------------------------------------

global function minIndex(sequence lst)
-- returns index of the lowest value from the list lst
-- if unsuccessful, returns 0
  atom mini

  for i = 1 to length(lst) do
    if sequence(lst[i]) then
      return 0
    elsif i=1 then
      mini = i
    elsif lst[i] < lst[mini] then
      mini = i
    end if
  end for
  return mini
end function

------------------------------------------------------------------------------

global function replaceString(sequence str, sequence old, sequence new)
-- replaces sub-sequence old by new in sequence str
  sequence s
  integer n, lg

  s = str
  lg = length(old)
  n = -1
  while n != 0 do
    n = match(old, s)
    if n > 1 then
      s = s[1..n-1]&new&s[n+lg..$]
    elsif n = 1 then
      s = new&s[n+lg..$]
    end if
  end while
  return s
end function

------------------------------------------------------------------------------

global function quote(sequence str)
  return "\"" & str & "\""
end function

------------------------------------------------------------------------------

global function unquote(sequence str)
  if (length(str) >= 2) and ( (str[1] ='"') and (str[$]='"') ) then
    return str[2..$-1]
  else
    return str
  end if
end function

------------------------------------------------------------------------------

global function splitString(sequence str, atom sep)
-- splits str string in tokens according to separator sep
  integer slen
  sequence result, s
  integer tmpStr

  result = {}
  slen = length(str)
  s = {}
  tmpStr = 0      -- si 1, on est entre deux guillemets

  for i = 1 to slen do
    if (str[i] = '"') and (sep != '"') then
      if tmpStr = 0 then
        tmpStr = 1
      else
        tmpStr = 0
      end if
	  s = s & str[i]
    else
      if tmpStr = 0 and str[i] = sep then
        result = append(result, trim(s))
        s = {}
 	  else
	    s = s & str[i]
	  end if
    end if
  end for
  result = append(result, trim(s))
  return result
end function

------------------------------------------------------------------------------

global function strtok(sequence str, atom sep)
-- splits str string in tokens according to separator sep
-- converts segments to atoms when needed
  sequence s

  s = splitString(str, sep)
  for i = 1 to length(s) do
    if isNumber(s[i]) or isHex(s[i]) then
      s[i] = val(s[i])
    elsif sequence(s[i]) then
      s[i] = unquote(s[i])
    end if
  end for
  return s
end function

------------------------------------------------------------------------------

global function tokenize(sequence tableau, atom sep)
-- assembles tokens in a string with separator sep
  sequence s

  if atom(tableau[1]) then
    s = sprintf("%d", tableau[1])
  else
    s = tableau[1]
  end if
  for i = 2 to length(tableau) do
    if atom(tableau[1]) then
      s &= {sep}&sprintf("%d", tableau[i])
    else
      s &= {sep}&tableau[i]
    end if
  end for
--  puts(f_debug, s&"\n")
  return s
end function

------------------------------------------------------------------------------

global function insert(object work, atom new, integer loc)
  integer workend
  if string(work) then
    workend = length(work)
    if loc > workend then
      work = work & new
      return work
    end if
    work = work[1..loc-1] & new & work[loc..workend]
  end if
  return work
end function

------------------------------------------------------------------------------

global function delete(object work, integer start, integer len)
  integer workend
  if string(work) then
    workend = length(work)
    if start > workend then return work end if
    if start > 1 then
	  work = work[1..start-1] & work[start+len..workend]
    else
      work = work[start+len..workend]
    end if
  end if
  return work
end function

------------------------------------------------------------------------------

global function replace(object work, sequence oldstring, sequence newstring)
  integer loc
  if string(work) then
    loc = match(oldstring,work)
    if loc > 0 then
    	 work = delete(work,loc,length(oldstring))
	 work = insert(work,newstring,loc)
    end if
  end if
  return work
end function

------------------------------------------------------------------------------

global function mid(sequence s, integer p, integer n)
  integer l

  l = length(s)
  if p > l then
    return s
  elsif n > l then
    return s[p..l]
  else
    return s[p..p+n-1]
  end if
end function

------------------------------------------------------------------------------

global function insertInSequence(object o, sequence s, integer p)
-- inserts object o in sequence s at position p
-- returns modified sequence
  if (p<1) or (p>length(s)) then return s end if
  if p = 1 then
    return o & s
  else
    return s[1..p-1] & o & s[p..$]
  end if
end function

------------------------------------------------------------------------------

global function removeFromSequence(sequence s, integer p)
-- deletes object at position p in sequence s
  if (p<1) or (p>length(s)) then return s end if
  if p = 1 then
    return s[2..$]
  elsif p = length(s) then
    return s[1..$-1]
  else
    return s[1..p-1] & s[p+1..$]
  end if
end function

------------------------------------------------------------------------------

global procedure hexDump(sequence s, integer output)
-- dumps a sequence in hex form to the specified output
  puts(output, "\n\nAddr  0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F\n")
  puts(output, "0000 ")
  for i = 1 to length(s) do
    puts(output, sprintf(" %02x", s[i]))
    if remainder(i, 16) = 0 then
      puts(output, sprintf("\n%04x ", floor(i/16)))
    end if
  end for
  puts(output, "\n\n")
end procedure

------------------------------------------------------------------------------

global procedure hexDump2(sequence s, integer output)
-- dumps a sequence in hex form to the specified output
  sequence hexa, str
  integer n, lg, offset

  lg = length(s)
  puts(output, "\n\nAddr  0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F\n")
  for i = 1 to floor(lg/16) do
    offset = (i-1)*16
    hexa = "00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00"
    str = "                "
    for j = 1 to 16 do
      if offset+j > lg then exit end if
      hexa[(j-1)*3+1..(j-1)*3+2] = sprintf("%02x", s[offset+j])
      if (s[offset+j] > 31) and (s[offset+j] < 255) then
        str[j] = s[offset+j]
      else
        str[j] = '.'
      end if
    end for
    printf(output, "%04x  %s  %s\n", {offset, hexa, str})
  end for
  puts(output, "\n\n")
end procedure

------------------------------------------------------------------------------

global function appendIfNew(sequence s, object o)
  if find(o, s) then
    return s
  else
    return append(s, o)
  end if
end function

-------------------------------------------------------------------------------

global function nbOccurences(sequence s, sequence item)
  integer nb, pos, lg

  nb = 0
  pos = 1
  lg = length(item)
  while pos > 0 do
    if match(item, s) then
      nb += 1
      s = delete(s, 1, pos+lg-1)
    end if
  end while
  return nb
end function

