-- myDebug.e
-- v1.0.1
-- 2013-03-11
-- debug controls

include file.e
include graphics.e
include image.e
include msgbox.e
include get.e
include myTypes.e
include wildcard.e

global integer
  f_debug,   -- debug file
  with_debug -- debug file

with_debug = 0
  
------------------------------------------------------------------------------

global function dateStamp(sequence msg)
-- prefixes the message by date and time for logging
  sequence cur_date

  cur_date = date()
  return sprintf("%d-%02d-%02d %02d:%02d:%02d -> %s\n", {
                             (cur_date[1] + 1900), cur_date[2], cur_date[3],
                              cur_date[4], cur_date[5], cur_date[6], msg
                            })
end function

------------------------------------------------------------------------------

global procedure logMsg(sequence msg)
-- records logs
  puts(f_debug, dateStamp(msg))
  flush(f_debug)
end procedure

------------------------------------------------------------------------------

function splitMessage(sequence msg, atom sep)
-- splits str string in tokens according to separator sep
  integer slen
  sequence result, s

  result = {}
  slen = length(msg)
  s = {}
  for i = 1 to slen do
    if msg[i] = sep then
      result = append(result, s)
      s = {}
 	else
	  s = s & msg[i]
	end if
  end for
  if length(s) > 0 then result = append(result, s) end if
  return result
end function

-------------------------------------------------------------------------------

procedure displayError(sequence msg)
-- prints a message in a frame at position pos
-- if pos is an atom (mostly 0), then center the frame in the screen
  sequence saved, vc, s
  integer h, w, x1, y1, x2, y2
  atom c

  text_color(BRIGHT_WHITE)
  bk_color(RED)
  vc = video_config()
  s = splitMessage(msg, '\n')
  h = length(s)
  w = 0
  for i = 1 to h do
    if length(s[i]) > w then w = length(s[i]) end if
  end for
  x1 = floor((vc[VC_COLUMNS]-w)/2)
  y1 = floor((vc[VC_LINES]-h)/2)
  x2 = x1+w+1
  y2 = y1+h
  saved = save_text_image({y1,x1}, {y2,x2})
  position(y1,x1)     puts(1, repeat(' ', w+2))
  position(y1+1,x1) puts(1, " "&msg&" ")
  position(y2,x1)     puts(1, repeat(' ', w+2))
  c = wait_key()
  display_text_image({y1,x1}, saved)
end procedure

------------------------------------------------------------------------------

global procedure warnError(sequence msg, atom quit)
-- logs an error and displays a warning
-- if quit is set then abort
  logMsg(msg)
  if platform() = WIN32 then
    void = message_box(msg, "Erreur", MB_OK+MB_ICONERROR)
  else
    displayError(msg)
  end if
  if quit then
    abort(1)
  end if
end procedure

------------------------------------------------------------------------------

global function showPrintable(object s)
-- returns only printable characters of a sequence
-- non printable characters are replace by a dot
  sequence res

  res = ""
  for i = 1 to length(s) do
    if integer(s[i]) then
      if (s[i] > 31) and (s[i] < 255) then
        res &= s[i]
      elsif s[i] = 9 then
        res &= "\\t"
      elsif s[i] = 13 then
        res &= "\\r"
      elsif s[i] = 10 then
        res &= "\\n"
      else
        res &= "."
      end if
    else
      res &= "."
    end if
  end for
  return res
end function

------------------------------------------------------------------------------

function translateVariable(sequence translator, sequence path)
  integer n
  
  n = path[$]
  for i = 1 to length(translator) do
    if length(translator[i]) != 2 then
      warnError("Each translation line must have 2 sequences: path & parameter values!", 0)
      return sprintf("%d",n)
    end if
    if wildcard_match(translator[i][1], path[1..$-1]) then
      if n <= length(translator[i][2]) then
        return translator[i][2][n]
      else
        return sprintf("%d",n)
      end if
    end if
  end for
  return sprintf("%d",n)
end function

------------------------------------------------------------------------------

procedure recurseSequence(object x, sequence Name, integer level,
                          sequence path, integer Output,
                          sequence translator)
-- prints a sequence structure in a human readable way
  integer subSequence
  sequence s, offset, prefix

  offset = ""
  prefix = ""
  if level > 0 then
    for i = 1 to level do offset &= ".  " end for
    if length(translator) > 0 then
      prefix = offset & "[" & translateVariable(translator, path) & "]"
    else
      prefix = offset & sprintf("[%d]", path[$])
    end if
  end if
  s = ""
  if Output = 0 then Output = 1 end if
  if integer(x) then
    s = sprintf("%s = %d", {Name, x})
    puts(Output, prefix&s&"\n")
  elsif atom(x) then
    if (x >= 0) and (x = floor(x)) then
      s = sprintf("%s = %.0f", {Name, x})
    elsif (x < 0) and (x = floor(x+1)) then
      s = sprintf("%s = %.0f", {Name, x})
    else
      s = sprintf("%s = %f", {Name, x})
    end if
    puts(Output, prefix&s&"\n")
  elsif string(x) then
    if length(x) = 0 then
      s = sprintf("%s = {}", {Name})
    else
      s = sprintf("%s = %s '%s'", {Name, sprint(x), showPrintable(x)})
    end if
    puts(Output, prefix&s&"\n")
  else
    subSequence = 0
    for i=1 to length(x) do
      if sequence(x[i]) then
        subSequence = 1
        exit
      end if
    end for
    if subSequence = 0 then
      s = sprintf("%s = %s", {Name, sprint(x)})
      puts(Output, prefix&s&"\n")
    else
      if length(s) = 0 then
        puts(Output, prefix&"\n")
      end if
      for i=1 to length(x) do
        recurseSequence(x[i], Name, level+1, append(path,i), Output, translator)
      end for
    end if
  end if
  if Output=f_debug then flush(f_debug) end if
end procedure

------------------------------------------------------------------------------

global procedure analyzeSequence(object x, sequence Name, integer Output, sequence translator)
  printf(Output, "%s", {Name})
  recurseSequence(x, "", 0, {}, Output, translator)
end procedure
