-- myConv.e
-- v1.0
-- 2010-11-13
-- conversion functions

-- function dec2str(atom n, atom d)
-- function base2str(atom n, atom b, atom d)
-- function num2str(atom n, atom b, atom d)
-- function str2num(sequence s, atom b)
-- function byte2word(atom l, atom h)
-- function highByte(atom x)
-- function lowByte(atom x)
-- function val(sequence str)
--   returns numerical value of string str
--   decimal separator may be a comma
--   displays a warning on error
-- function atom2string(atom val, sequence fmt)
--   converts numerical value val according to format fmt
--   replaces decimal separator by a comma (for non english speaking countries)
-- function object2string(object s)
--   converts an object in a sequence
--   sub-sequences appear between curly brackets as with print()
--   sub-sequence items are separated by commas as with print()
-- function utf8_to_ascii(sequence s)
--   created from Tommy's Unicode Library 1.2
--   copyright (c) 2003-2004 Tommy Carlier
-- function hexString(sequence s)
--   returns hexadecimal representation of a sequence
-- function dos2unix(sequence s)
--   converts DOS end of lines to Unix ones

include myTypes.e
include myDebug.e

-------------------------------------------------------------------------------

global function dec2str(atom n, atom d)
  sequence result

  result = repeat(32, d)
  for i = d to 1 by -1 do
    result[i] = remainder(n, 10) + 48
    n = floor(n / 10)
  end for
  return result
end function

-------------------------------------------------------------------------------

global function base2str(atom n, atom b, atom d)
  atom c
  sequence result

  result = repeat(32, d)
  c = floor((log(b)/log(2))+0.5)
  for i = d to 1 by -1 do
    result[i] = DIGITS[and_bits(n / power(2, (d-i)*c), b-1)+1]
  end for
  return result
end function

-------------------------------------------------------------------------------

global function num2str(atom n, atom b, atom d)

  if b = 2 then
    if d>32 then d = 32 end if
    return base2str(n, 2, d)
  elsif b = 8 then
    if d>11 then d = 11 end if
    return base2str(n, 8, d)
  elsif b = 10 then
    return dec2str(n, d)
  elsif b = 16 then
    if d>8 then d = 8 end if
    return base2str(n, 16, d)
  else
    return {}
  end if
end function

-------------------------------------------------------------------------------

global function str2num(sequence s, atom b)
  atom l

  l = 0
  for i = 1 to length(s) do
    l = l + (match({s[i]}, DIGITS)-1) * floor(power(b, length(s)-i))
  end for
  return l
end function

-------------------------------------------------------------------------------

global function byte2word(atom l, atom h)
  return h*#100+l
end function

-------------------------------------------------------------------------------

global function highByte(atom x)
  return floor(x / #100)
end function

-------------------------------------------------------------------------------

global function lowByte(atom x)
  return remainder(x, #100)
end function

------------------------------------------------------------------------------

global function val(sequence str)
-- returns numerical value of string str
-- decimal separator may be a comma
-- displays a warning on error
  integer n
  sequence s

  if length(str) = 0 then
    return 0
  end if
  n = find(',', str)
  if n>0 then str[n] = '.' end if
  s = value(str)
  if s[1] = GET_SUCCESS then
    return s[2]
  else
    return 0
--    warnError(sprintf("Erreur de conversion: '%s'", {str}), 1)
  end if
end function

------------------------------------------------------------------------------

global function atom2string(atom val, sequence fmt)
-- converts numerical value val according to format fmt
-- replaces decimal separator by a comma (for non english speaking countries)
  integer n
  sequence s

  s = sprintf(fmt, val)
  n = find('.', s)
  if n>0 then s[n] = ',' end if
  return s
end function

------------------------------------------------------------------------------

global function object2string(object s)
-- converts an object in a sequence
-- sub-sequences appear between curly brackets as with print()
-- sub-sequence items are separated by commas as with print()
  sequence res

  if integer(s) then
    res = sprintf("%d", s)
  elsif atom(s) then
    if (s >= 0) and (s = floor(s)) then
      res = sprintf("%.0f", s)
    elsif (s < 0) and (s = floor(s+1)) then
      res = sprintf("%.0f", s)
    else
      res = sprintf("%f", s)
    end if
  elsif string(s) then
    res = "\"" & s &  "\""
  else
    res = {123}  -- {
    for i=1 to length(s) do
      res &= object2string(s[i])&","
    end for
    res[$] = 125   -- }
  end if
  return res
end function

------------------------------------------------------------------------------

global function utf8ToAscii(sequence s)
  integer i, lg, byte, bytes
  sequence res
  atom char

  i = 1
  res = ""
  lg = length(s)
  if lg < 2 then return s end if
  while i <= lg do
    bytes = 1
    char = s[i]
    if and_bits(s[i],#80) = #00 then
      char = and_bits(s[i], #7F)
    elsif ((i+1) <= lg)  and   -- else Unexpected end of UTF-8 string
          (and_bits(s[i],  #E0) = #C0) and
          (and_bits(s[i+1],#C0) = #80) then
      if and_bits(s[i],#FE) != #C0 then   -- else Overlong UTF-8 char
        bytes = 2
        char = and_bits(s[i], #1F)
      end if
    elsif ((i+2) <= lg)  and   -- else Unexpected end of UTF-8 string
          (and_bits(s[i],  #F0) = #E0) and
          (and_bits(s[i+1],#C0) = #80) and
          (and_bits(s[i+2],#C0) = #80) then
      if (and_bits(s[i],  #FF) != #E0) or
           (and_bits(s[i+1],#E0) != #80) then   -- else Overlong UTF-8 char
        bytes = 3
        char = and_bits(s[i], #0F)
      end if
    elsif ((i+3) <= lg)  and   -- else Unexpected end of UTF-8 string
          (and_bits(s[i],  #F8) = #F0) and
          (and_bits(s[i+1],#C0) = #80) and
          (and_bits(s[i+2],#C0) = #80) and
          (and_bits(s[i+3],#C0) = #80) then
      if (and_bits(s[i],  #FF) != #F0) or
           (and_bits(s[i+1],#F0) != #80) then   -- else Overlong UTF-8 char
        bytes = 4
        char = and_bits(s[i], #07)
      end if
    elsif ((i+4) <= lg)  and   -- else Unexpected end of UTF-8 string
          (and_bits(s[i],  #FC) = #F8) and
          (and_bits(s[i+1],#C0) = #80) and
          (and_bits(s[i+2],#C0) = #80) and
          (and_bits(s[i+3],#C0) = #80) and
          (and_bits(s[i+4],#C0) = #80) then
      if (and_bits(s[i],  #FF) != #F8) or
         (and_bits(s[i+1],#F8) != #80) then   -- else Overlong UTF-8 char
        bytes = 5
        char = and_bits(s[i], #03)
      end if
    elsif ((i+5) <= lg) and  -- else Unexpected end of UTF-8 string
          (and_bits(s[i],  #FE) = #FC) and
          (and_bits(s[i+1],#C0) = #80) and
          (and_bits(s[i+2],#C0) = #80) and
          (and_bits(s[i+3],#C0) = #80) and
          (and_bits(s[i+4],#C0) = #80) and
          (and_bits(s[i+5],#C0) = #80) then
      if (and_bits(s[i],  #FF) != #FC) or
         (and_bits(s[i+1],#FC) != #80) then   -- else Overlong UTF-8 char
        bytes = 6
        char = and_bits(s[i], #01)
      end if
    end if
    if bytes > 1 then
      for n=1 to bytes-1 do
        byte = and_bits(s[i+n],#3F)
        char = (char*64)+byte
      end for
      if ((char>=#D800) and (char<=#DFFF)) or
         (remainder(char,#100)>=#FE) then
        res &= s[i]
        for n=1 to bytes-1 do
          res &= s[i+n]
        end for
      else
        res &= char
      end if
      i += bytes-1
    else
      res &= char
    end if
    i += 1
  end while
  return res
end function

------------------------------------------------------------------------------

global function isoToUtf8(sequence s)
  sequence res, bits, u8

  res = ""
  for i = 1 to length(s) do
    if s[i] > 65536 then     -- 11110xxx 10xxxxxx 10xxxxxx 10xxxxxx
      bits = reverse(int_to_bits(s[i],21))
      u8 = {{1,1,1,1,0}&bits[1..3],{1,0}&bits[4..9],{1,0}&bits[10..15],{1,0}&bits[16..21]}
      res &= bits_to_int(reverse(u8[1]))&bits_to_int(reverse(u8[2]))&bits_to_int(reverse(u8[3]))&bits_to_int(reverse(u8[4]))
   elsif s[i] > 2048 then  -- 1110xxxx 10xxxxxx 10xxxxxx
      bits = reverse(int_to_bits(s[i],16))
      u8 = {{1,1,1,0}&bits[1..4],{1,0}&bits[5..10],{1,0}&bits[11..16]}
      res &= bits_to_int(reverse(u8[1]))&bits_to_int(reverse(u8[2]))&bits_to_int(reverse(u8[3]))
    elsif s[i] > 128 then   -- 110xxxxx 10xxxxxx
      bits = reverse(int_to_bits(s[i],11))
      u8 = {{1,1,0}&bits[1..5],{1,0}&bits[6..11]}
      res &= bits_to_int(reverse(u8[1]))&bits_to_int(reverse(u8[2]))
    else                -- 0xxxxxxx
      res &= s[i]
    end if
  end for
  return res
end function

------------------------------------------------------------------------------

global function hexString(sequence s)
-- returns hexadecimal representation of a sequence
  sequence result

  result = ""
  for i = 1 to length(s) do
    result &= sprintf("%02x", s[i])
  end for
  return result
end function

------------------------------------------------------------------------------

global function dos2unix(sequence s)
-- converts DOS end of lines to Unix ones
  sequence t
  integer l

  t = {}
  l = match({13,10}, s)
  while l > 0 do
    t = t & s[1..l-1]
    s = s[l+1..length(s)]
    l = match({13,10}, s)
  end while
  t = t & s
  return t
end function


