include get.e
include machine.e
include msgbox.e

-- v0.1 March 22, 2007
-- initial release

-- v0.2 March 24, 2007
-- putXXX returns checksum

-- v0.3 March 27, 2007
-- putString completely rewritten

-- v0.4 April 12, 2011
-- corrected error in putDate (time checking)

-- v0.5 August 15, 2012
-- added fixed and variable size string fields

global atom mem4, mem8
mem4 = allocate(4)
mem8 = allocate(8)

global function getSignature(integer fn)
-- read a 6 byte signature at current position in database file
  sequence s

  s = get_bytes(fn, 6)
  if length(s) < 6 then   -- end of file
    return {GET_EOF, ""}
  else
    return {GET_SUCCESS, s}
  end if
end function

global function getUInt8(integer fn)
-- read a single unsigned byte at current position in database file
  integer i

  i = getc(fn)
  if i = -1 then   -- end of file
    return {GET_EOF, 0}
  else
    return {GET_SUCCESS, i}
  end if
end function

global function getSInt8(integer fn)
-- read a single signed byte at current position in database file
  integer i

  i = getc(fn)
  if i = -1 then   -- end of file
    return {GET_EOF, 0}
  else
    if i >= #80 then
      return {GET_SUCCESS, i-#100}
    else
      return {GET_SUCCESS, i}
    end if
  end if
end function

global function getUInt16(integer fn)
-- read a 2 byte unsigned word at current position in database file
  sequence s

  s = get_bytes(fn, 2)
  if length(s) < 2 then   -- end of file
    return {GET_EOF, 0}
  else
    return {GET_SUCCESS, s[1] + s[2]*#100}
  end if
end function

global function getSInt16(integer fn)
-- read a 2 byte signed word at current position in database file
  integer i
  sequence s

  s = get_bytes(fn, 2)
  if length(s) < 2 then   -- end of file
    return {GET_EOF, 0}
  else
    i = s[1] + s[2]*#100
    if i >= #8000 then
      return {GET_SUCCESS, i - #10000}
    else
      return {GET_SUCCESS, i}
    end if
  end if
end function

global function getUInt32(integer fn)
-- read a 4 byte unsigned integer at current position in database file
  sequence s

  s = get_bytes(fn, 4)
  if length(s) < 4 then   -- end of file
    return {GET_EOF, 0}
  else
    return {GET_SUCCESS, bytes_to_int(s)}
  end if
end function

global function getSInt32(integer fn)
-- read a 4 byte signed integer at current position in database file
  atom a
  sequence s

  s = get_bytes(fn, 4)
  if length(s) < 4 then   -- end of file
    return {GET_EOF, 0}
  else
    a = bytes_to_int(s)
    if a >= #80000000 then
      return {GET_SUCCESS, a - #100000000}
    else
      return {GET_SUCCESS, a}
    end if
  end if
end function

global function getUInt64(integer fn)
-- read a 8 byte long integer at current position in database file
  sequence s

  s = get_bytes(fn, 8)
  if length(s) < 8 then   -- end of file
    return {GET_EOF, 0.0}
  else
    return {GET_SUCCESS, bytes_to_int(s[1..4]) + bytes_to_int(s[5..8]*#10000)}
  end if
end function

global function getFloat32(integer fn)
-- read a 4 byte float at current position in database file
  sequence s

  s = get_bytes(fn, 4)
  if length(s) < 4 then   -- end of file
    return {GET_EOF, 0.0}
  else
    return {GET_SUCCESS, float32_to_atom(s)}
  end if
end function

global function getFloat64(integer fn)
-- read a 8 byte float at current position in database file
  sequence s

  s = get_bytes(fn, 8)
  if length(s) < 8 then   -- end of file
    return {GET_EOF, 0.0}
  else
    return {GET_SUCCESS, float64_to_atom(s)}
  end if
end function

global function getString(integer fn)
-- read a variable size 0-terminated string at current position in database file
  sequence s
  integer n

  s = {}
  while 1 do
    n = getc(fn)
    if n = -1 then return {GET_EOF, ""} elsif n = 0 then exit end if
    s = append(s, n)
  end while
  return {GET_SUCCESS, s}
end function

global function getStringField(integer fn, integer size)
-- read a fixed size 0-terminated string field at current position in database file
  sequence s
  integer n

  s = get_bytes(fn, size)
  if length(s) < size then   -- end of file
    return {GET_EOF, ""}
  else
    n = find(0, s)
    if n > 1 then
      return {GET_SUCCESS, s[1..n-1]}
    else
      return {GET_SUCCESS, ""}
    end if
  end if
end function

global function getDate(integer fn)
-- read a 7 byte date at current position in database file
  sequence s, dt

  s = get_bytes(fn, 7)
  dt = {}
  if length(s) < 7 then   -- end of file
    return {GET_EOF, {0,0,0,0,0,0}}
  else
    dt = append(dt, getUInt16(fn))
    for i = 1 to 5 do dt = append(dt, getUInt8(fn)) end for
    return {GET_SUCCESS, dt}
  end if
end function

function cksum(sequence s)
  integer cks

  cks= 0
  for i = 1 to length(s) do
    cks += s[i]
  end for
  return cks
end function

global function putSignature(integer fn, sequence s)
-- ok = put a 6 byte signature at current position in database file
  if length(s) > 6 then return 0 end if
  puts(fn, s)
  return cksum(s)
end function

global function putInt8(integer fn, integer i)
-- ok = put a single signed/unsigned byte at current position in database file
  if i > #100 then return 0 end if
  puts(fn, i)
  return i
end function

global function putInt16(integer fn, integer i)
-- ok = put a 2 byte signed/unsigned word at current position in database file
  integer l, h

  if i > #10000 then return 0 end if
  l = remainder(i, #100)
  h = floor(i / #100)
  puts(fn, {l, h})
  return l+h
end function

global function putInt32(integer fn, atom a)
-- ok = put a 4 byte signed/unsigned integer at current position in database file
  sequence s

  if a > #100000000 then return 0 end if
  poke4(mem4, a)
  s = peek({mem4, 4})
  puts(fn, s)
  return cksum(s)
end function

global function putInt64(integer fn, atom a)
-- ok = put a 8 byte signed/unsigned long integer at current position in database file
  sequence s

  if a > #10000000000000000 then return 0 end if
  poke4(mem8, remainder(a, #100000000))
  poke4(mem8+4, floor(a / #100000000))
  s = peek({mem8, 8})
  puts(fn, s)
  return cksum(s)
end function

global function putFloat32(integer fn, atom a)
-- ok = put a 4 byte float at current position in database file
  sequence s

  s = atom_to_float32(a)
  puts(fn, s)
  return cksum(s)
end function

global function putFloat64(integer fn, atom a)
-- ok = put a 8 byte float at current position in database file
  sequence s

  s = atom_to_float64(a)
  puts(fn, s)
  return cksum(s)
end function

function Complete(sequence s, atom len)
--Return a string right-padded to length len

  if length(s) >= len then
    return s
  else
    return s & repeat(0, len-length(s))
  end if
end function

global function putString(integer fn, sequence s)
-- put a variable size 0-terminated string at current position in database file
  sequence buffer

  buffer = s&0
  puts(fn, buffer)
  return cksum(buffer)
end function

global function putStringField(integer fn, sequence s, integer size)
-- ok = put a fixed size 0-terminated string field at current position in database file
  sequence buffer

  if length(s) > size then   -- string too long, truncate
    buffer = s[1..size-1]&0
  else
    buffer = Complete(s, size)
  end if
  puts(fn, buffer)
  return cksum(buffer)
end function

global function putDate(integer fn, integer year, integer month, integer day, integer hours, integer minutes, integer seconds)
-- ok = put a 7 byte date at current position in database file
  integer cks
  
  cks = 0
  if year >=0 then cks += putInt16(fn, year) else return 0 end if
  if (month > 0) and (month <= 12) then cks += putInt8(fn, month) else return 0 end if
  if (day > 0) and (month <= 31) then cks += putInt8(fn, day) else return 0 end if
  if (hours >= 0) and (hours < 24) then cks += putInt8(fn, hours) else return 0 end if
  if (minutes >= 0) and (minutes < 60) then cks += putInt8(fn, minutes) else return 0 end if
  if (seconds >= 0) and (seconds < 60) then cks += putInt8(fn, seconds) else return 0 end if
  return cks
end function

