-- MakeRes  v.1.1
-- Build a .RES (resource) file for a Euphoria program.
--
-- (c) david cuny
-- e-mail: dcuny@lanset.com

--
-- input:   <file>[.ex|.e|.exw]
-- output:  <file>.res
--
-- read the input file, and all it's includes for the strings:
--
--  "rOpen("
--  "rFile("
--  "rReadBmp("
--
-- and extracts the file names.
--
-- It then builds a .RES file in the form:
--
-- <num files>  <name1> <length1> <data1>  <name2> ...
--
-- where:
--                                         
-- <numfiles>   is the count of all files
-- <name>       is the name of the file, ending with a "\n";
-- <length>     is the length of the file, ending with a "\n"; and
-- <data>       is the contents of the file
--
-- for example, the header for this file might be:
--   "makeres.ex\n3421\n"
--

include file.e
include get.e
include wildcard.e

with trace

sequence 
    includedFiles,      -- files already included
    resFiles            -- resource files used
    
    includedFiles = { "res.e" }     -- don't include
    resFiles = {}

    
-----------------------------------------------------------------------------
function butLast( sequence s )
    -- return everything but last item
    return s[1..length(s)-1]
end function


-----------------------------------------------------------------------------
constant
    WhiteSpace = " \t\n"
-----------------------------------------------------------------------------
function extractWordAfter( sequence pattern, sequence s )

    -- return word following given position
    -- handle quotes                       
    
    integer loopFlag, quoteFlag, specialFlag, at
    sequence word

    -- find pattern
    at = match( pattern, s ) + length( pattern )

    -- skip spaces following...    
    for i = at to length( s ) do
        if find( s[i], WhiteSpace ) then
            -- skip
            at = i + 1
        else
            -- start of word
            exit
        end if
    end for


    -- read until non-space
    word = ""
    loopFlag = 1
    quoteFlag = 0
    specialFlag = 0
    
    
    while loopFlag do
    
        -- special characters?
        if specialFlag  then  
            if    s[at] = 'n'   then s[at] = '\n'
            elsif s[at] = 'r'   then s[at] = '\r'
            elsif s[at] = 't'   then s[at] = '\t'
            elsif s[at] = '\\'  then s[at] = '\\'
            elsif s[at] = '"'   then s[at] = '"'
            elsif s[at] = '\''  then s[at] = '\''
            end if   
            
            specialFlag = 0
            
        elsif s[at] = '\\' then
            specialFlag = 1
            s[at] = 0

        elsif quoteFlag then
            if s[at] = '"' then   
                quoteFlag = 0
                loopFlag = 0
            end if          
            
        elsif s[at] = '"' then
            quoteFlag = 1
            
        elsif find( s[at], WhiteSpace ) then
            loopFlag = 0
            
            -- end of line?
            if s[at] = '\n' then
                -- ignore
                s[at] = 0
            end if
            
        end if

        -- accumulate
        if s[at] then
            word = word & s[at]
        end if
        
        -- move ahead
        at = at + 1  
        
        -- end of line?
        if at > length( s ) then
            -- exit
            loopFlag = 0
        end if
            
    end while

    return lower( word )
    
end function


constant EXT = { "", ".EX", ".EXW", ".E", ".EW" }
-----------------------------------------------------------------------------
function getInFileName()
    -- get the file name from the command line
    -- if none is found, ask the user for it
    
    sequence args, fName

    -- name is unknown
    fName = {}

    -- get the command line
    args = command_line()
    if length( args ) > 2 then
        fName = args[3]
    end if

    if length( fName ) = 0 then
        
        -- get prompt from user
        puts( 1, "File to link? " )
        fName = gets(0)
        puts( 1, "\n" )
        
        -- remove line feed
        fName = butLast( fName )
        
        -- empty?
        if length( fName ) = 0 then
            abort(0)
        end if   
    
    end if
    
    -- did they remember an extention?
    for i = 1 to length( EXT ) do
        if compare( dir( fName & EXT[i] ), -1 ) != 0 then
            fName = fName & EXT[i]
            exit
        end if
    end for
    
    return fName
             
end function            



-----------------------------------------------------------------------------
function getOutFileName( sequence s )
    
    -- convert the input file name into an output file name
    integer at
    
    -- outfile
    at = find( '.', s)
    if at = 0 then
        at = length( s )
    end if      

    -- replace with "res"
    return s[1..at] & "RES"
    
end function


constant
    EuPlace = getenv( "EUDIR" ),
    Place = { "", EuPlace & "\\", EuPlace & "\\INCLUDE\\" }
-----------------------------------------------------------------------------
function findFile( sequence fName )

    -- returns where a file is
    -- looks in the usual places
    
    -- look in the usual places
    for i = 1 to length( Place ) do
        if sequence( dir( Place[i] & fName ) ) then
            return Place[i] & fName
        end if
    end for
    
    printf( 1, "Unable to locate file %s.\n", {fName} )
    abort(0)
    
end function

-----------------------------------------------------------------------------
procedure includeFile( sequence fName, integer deep )

    -- open an include file for processing
    integer inFile, line
    sequence text
    object data
    
    -- get the file
    fName = findFile( fName )
    
    -- open if
    inFile = open( fName, "r" )
    if inFile = -1 then
        printf( 1, "Error opening file %s.\n", {fName} )
        abort(0)
    end if

    -- line number
    line = 0

    while 1 do   
    
        -- get text
        data = gets( inFile )
        
        -- end of file?
        if integer( data ) then
            exit
        end if
    
        -- increment line
        line = line + 1
    
        -- look for include
        if match( data, "include " ) = 1
        or match( data, " include " ) then

            -- get file name
            text = extractWordAfter( "include ", data )

            -- new?
            if not find( text, includedFiles ) then
            
                -- add to included list
                includedFiles = append( includedFiles, text )
                
                -- include file
                includeFile( text, deep + 1 ) 
                
            end if
            
            
        elsif match( "rOpen(", data ) = 1
        or    match( " rOpen(", data ) 
        or    match( "rFile(", data ) = 1
        or    match( " rFile(", data )
        or    match( "rReadBmp(", data ) = 1
        or    match( " rReadBmp(", data ) then
        
            -- handle resFile

            -- which key word?
            if match( "rOpen(", data ) then
                -- get file name from rOpen(...
                text = extractWordAfter( "rOpen(", data )
                
            elsif match( "rFile(", data ) then
                -- get file name from rFile(...            
                text = extractWordAfter( "rFile(", data )
                
            elsif match( "rReadBmp(", data ) then
                -- get file name from rBitMap(...            
                text = extractWordAfter( "rReadBmp(", data )
                
            end if
            
            -- quoted?
            if text[1] = '"' then

                -- remove quotes            
                text = text[2..length(text)-1]
                
                -- new?
                if not find( text, resFiles ) then
                
                    -- add to included list
                    resFiles = append( resFiles, text )
                    
                end if
            end if            
        end if
    
    end while                       
    
    close( inFile )

end procedure



integer inFile, outFile
sequence inName, outName
object data

inName  = getInFileName()
outName = getOutFileName( inName )

includeFile( inName, 0 )

outFile = open( outName, "wb" )

-- write out the count
printf( outFile, "%d\n", length( resFiles ) )

-- create the res files
for i = 1 to length( resFiles ) do

    -- status
    printf( 1, "%d\tAppending %s ...", {i,resFiles[i]} )

    -- put the file name in the header
    puts( outFile, resFiles[i] & "\n" )
    
    -- get file information
    data = dir( resFiles[i] )
    if length( data ) = 0 then
        puts( 1, "ERROR: Can't locate file!\n" )
        abort(0)
    end if
    
    -- the size
    data = data[1][D_SIZE]

    -- put length in header
    printf( outFile, "%d\n", {data} ) 
    
    -- open the file for reading
    inFile = open( resFiles[i], "rb" )
    
    -- write the file
    for j = 1 to data do
        puts( outFile, getc( inFile ) )
    end for

    -- close the file
    close( inFile )
    puts( 1, "ok\n" )
    
end for

-- close the res file
close( outFile )
    
puts( 1, "\n" )
printf( 1, "Complete. Created file %s\n", {lower(outName)} )



