---------------------------------------------------------------------------------
-- heaps.e
-- version 1.0
-- written by Chris Bensler : bensler@mail.com
-- LAST MODIFIED : 08/11/02 / 19:12
---------------------------------------------------------------------------------
-- prev_addr, next_addr, ptype, allocated_memory
--{prev,next,type}

---------------------------------------------------------------------------------
-- Defines --
---------------------------------------------------------------------------------
constant
   HEAP     = #FFFFFFFE
  ,PTR      = #FFFFFFFF

atom lpMain     lpMain    = machine_func(16, 24)+12
   poke4(lpMain-12, {lpMain, lpMain, HEAP, lpMain, lpMain, lpMain})

atom lpActive   lpActive  = lpMain
atom addr, tmp, prev, next



---------------------------------------------------------------------------------
-- Local Routines --
---------------------------------------------------------------------------------

------------------------< clear_heap() >------------------------
-- Deallocates and releases all child heaps and pointers,
--  without deallocating the given heap.
----------------------------------------------------------------
procedure clear_heap(atom p)
 atom n, a
   n = peek4u(p+4)
   while n != p do
      a = n
      n = peek4u(a+4)
      if peek4u(a+8) = HEAP then
         clear_heap(a+12)
      end if
      machine_proc(17, a)
   end while
end procedure
----------------------------------------------------------------



---------------------------------------------------------------------------------
-- Allocation Routines --
---------------------------------------------------------------------------------

------------------------< alloc() >------------------------
-- Allocate i contingous bytes of memory.
-- Returns a pointer to the allocated memory.
----------------------------------------------------------------
global function alloc(integer l) -- 16
   addr = machine_func(16, l+12)
   next = peek4u(lpActive+4)
   poke4(lpActive+4,addr)  -- prev:next
   poke4(addr,lpActive)    -- this:prev
   poke4(addr+4,next)      -- this:next
   poke4(addr+8,PTR)       -- this:type
   poke4(next, addr)       -- next:prev
   return addr+12
end function
----------------------------------------------------------------


------------------------< alloc_sz() >------------------------
-- Allocates s as a null terminated string ( s & 0 ) in memory and
--  pokes it to the new memory location.
-- Returns a pointer to the allocated string.
----------------------------------------------------------------
global function alloc_lpsz(sequence sz) -- 16 sz
   addr = machine_func(16, length(sz)+1+12)
   next = peek4u(lpActive+4)
   poke4(lpActive+4,addr)  -- prev:next
   poke4(addr,lpActive)    -- this:prev
   poke4(addr+4,next)      -- this:next
   poke4(addr+8,PTR)       -- this:type
   poke4(next,addr)        -- next:prev
   tmp = addr+12
   poke(tmp,sz) -- poke the string
   poke(tmp+length(sz),0) -- poke the null terminator
   return tmp
end function
----------------------------------------------------------------


------------------------< alloc_heap() >------------------------
-- Allocate a new heap as a child of heap:a1.
-- a2 will automatically be set as the active heap.
----------------------------------------------------------------
global function alloc_heap(atom a) -- 16
-- {prev, next, type, paren, child}
   a -= 12
   addr = machine_func(16, 24)
   next = peek4u(a+4)
   poke4(a+4,addr)         -- prev:next
   poke4(addr,a)           -- this:prev
   poke4(addr+4,next)      -- this:next
   poke4(addr+8, HEAP)     -- this:type
   poke4(addr+16,addr+12)  -- this:next
   poke4(addr+20,a)        -- this:paren
   poke4(next, addr)       -- next:prev
   lpActive = addr+12
   return addr+24
end function
----------------------------------------------------------------




---------------------------------------------------------------------------------
-- Deallocation Routines --
---------------------------------------------------------------------------------

------------------------< dealloc() >------------------------
-- Deallocates the pointer a, which was allocated using alloc() or alloc_sz()
----------------------------------------------------------------
global procedure dealloc(atom a) -- 17
   addr = a-12
   prev = peek4u(addr)     -- this:prev
   next = peek4u(addr+4)   -- this:next
   poke4(next,prev)        -- next:prev
   poke4(prev+4,next)      -- prev:next
   machine_proc(17, addr)
end procedure
----------------------------------------------------------------


------------------------< dealloc_heap() >------------------------
-- Deallocates and releases the specified heap and all child heaps and pointers.
----------------------------------------------------------------
global procedure dealloc_heap(atom h) -- 17
 atom n, a
   h -= 12
   prev = peek4u(h-12)
   next = peek4u(h-8)
   n = peek4u(h+4)
   while n != h do
      a = n
      n = peek4u(a+4)
      if peek4u(a+8) = HEAP then
         clear_heap(a+12)
      end if
      machine_proc(17, a)
   end while
   poke4(next,prev)
   poke4(prev+4,next)
   lpActive = peek4u(h+8)
   machine_proc(17, h-12)
end procedure
----------------------------------------------------------------


------------------------< dealloc_all() >------------------------
-- Deallocates and releases all heaps and pointers.
----------------------------------------------------------------
global procedure dealloc_all()
 atom n, a
   n = peek4u(lpMain+4)
   while n != lpMain do
      a = n
      n = peek4u(a+4)
      if peek4u(a+8) = HEAP then
         clear_heap(a+12)
      end if
      machine_proc(17, a)
   end while
   poke4(lpMain+4, lpMain)
end procedure
----------------------------------------------------------------



---------------------------------------------------------------------------------
-- Heap Management Routines --
---------------------------------------------------------------------------------

------------------------< set_active_heap() >------------------------
-- Set the specified heap as the active heap.
-- The active heap is used as the parent for all successive heaps and pointers.
----------------------------------------------------------------
global procedure set_active_heap(atom a)
   lpActive = a - 12
end procedure
----------------------------------------------------------------

------------------------< active_heap() >------------------------
-- Returns a pointer to the active heap.
----------------------------------------------------------------
global function active_heap()
   return lpActive + 12
end function
----------------------------------------------------------------

------------------------< main_heap() >------------------------
-- Returns a pointer to the main heap.
-- The main heap is used as the parent to ALL heaps and pointers.
-- You can allocate pointers directly to the main heap,
--  by setting it active (default, until changed)
----------------------------------------------------------------
global function main_heap()
   return lpMain + 12
end function
----------------------------------------------------------------



---------------------------------------------------------------------------------
-- HISTORY --

-------------------------------------------------
-- version 1.0 --
-- simply reversioned from 0.17a

-------------------------------------------------
-- version 0.17a --
-- changed the type value to 4bytes, instead of 1
--  and changed the HEAP type and PTR type constants
--   to allow for storage of struct references

-------------------------------------------------
-- version 0.16a --
-- didn't like my last changes, reverted to version v0.14a
-- structs should be handled directly from structs.e
-- renamed:
--    get_active_heap()    --> active_heap()
-- added:
--    function main_heap()

-------------------------------------------------
-- version 0.15a --
-- renamed:
--    alloc_sz()     --> alloc_lpsz()
-- added:
--    dealloc_lpsz(): equivalent to dealloc(), for symmetry with alloc_lpsz()
--    alloc_struct()
--    dealloc_struct()
--    structs.e handshake system

-------------------------------------------------
-- version 0.14a -- functionally stable
-- globals:
--    function alloc()
--    function alloc_sz()
--    function alloc_heap()
--    procedure dealloc_heap()
--    procedure dealloc()
--    procedure dealloc_all()
--    procedure set_active_heap()
--    function get_active_heap()
