-----------------------------------------------------
--! Perlin Noise Lib                              !--
--! Version 0.5.1                                 !--
--! Core Library                                  !--
--! Version 0.5.2                                 !--
--!-----------------------------------------------!--
--! Working Beta                                  !--
--! Date: 26th August 2003                        !--
--! Daniel McGrath                                !--
--!-----------------------------------------------!--
--! Notes:                                        !--
--! Tiling in the z axis is not supported atm for !--
--! 3D Noise                                      !--
--!-----------------------------------------------!--
--! Here are just some of the planned features:   !--
--!                                               !--
--! Image Script files. To create the textures.   !--
--!                                               !--
--! Seperate Noise channels for each of the RGB   !--
--! Channels. Maybe even a 'A' channel            !--
--!                                               !--
--! Seperate colour schemes for each layer,       !--
--! giving a FAR greater control over the final   !--
--! texture.                                      !--
--!                                               !--
--! More advanced functions for manipulating      !--
--! sound for PSound.ew                           !--
-----------------------------------------------------

include Lib\PSupport.e

global sequence PNL_RandList  -- Contains the random data used for make each layer.
integer PNL_IFS               -- Interpolate function switch. 0 = Linear. 1 = Cosine.
integer PNL_NTO               -- Noise Tile option. 0 = infinite. 1 = tile
integer PNL_NSO               -- No Smoothing Option. 0 = Smoothing. 1 = No-Smoothing
integer PNL_SIZE_x            -- Length of the generated Noise patterns. Must be > 0.
integer PNL_SIZE_y            -- Height of the generated Noise patterns. Must be > 0.
integer PNL_SIZE_z            -- Depth of the generated Noise patterns. Must be > 0.
integer PNL_SAID              -- Routine ID for the function that generates Amplitudes
integer PNL_CCID              -- Routine ID for the function that converts a Perlin value to a colour

------------------------------------------------------------------------------------------
--------------------------------- INITIALISE TO DEFAULTS ---------------------------------
------------------------------------------------------------------------------------------
PNL_IFS = 0
PNL_NTO = 0
PNL_NSO = 1
PNL_SIZE_x = 128
PNL_SIZE_y = 128
PNL_SIZE_z = 32
PNL_SAID = routine_id("PNL_S_AMP")
PNL_CCID = routine_id("PNL_Color_Grey")
------------------------------------------------------------------------------------------
------------------------------ end of INITIALISE TO DEFAULTS -----------------------------
------------------------------------------------------------------------------------------

------------------------------------------------------------------------------------------
----------------------------------- INTERFACE Section ------------------------------------
------------------------------------------------------------------------------------------

global procedure Set_x(atom x)
    PNL_SIZE_x = x
end procedure

global function Read_x()
    return PNL_SIZE_x
end function

global procedure Set_y(atom y)
    PNL_SIZE_y = y
end procedure

global function Read_y()
    return PNL_SIZE_y
end function

global procedure Set_z(atom z)
    PNL_SIZE_z = z
end procedure

global function Read_z()
    return PNL_SIZE_z
end function

global procedure Set_IFS(atom x)
    PNL_IFS = x
end procedure

global function Read_IFS()
    return PNL_IFS
end function

global procedure Set_NTO(atom x)
    PNL_NTO = x
end procedure

global function Read_NTO()
    return PNL_NTO
end function

global procedure Set_NSO(atom x)
    PNL_NSO = x
end procedure

global function Read_NSO()
    return PNL_NSO
end function

global function Set_SAID(object x)
    atom SAID

    SAID = PNL_SAID
    if sequence(x) then
        PNL_SAID = routine_id(x)
    else
        PNL_SAID = x
    end if

    if PNL_SAID = -1 then
        PNL_SAID = SAID
        return 0
    end if

    return 1
end function

global function Read_SAID()
    return PNL_SAID
end function

global function Set_CCID(sequence x)
    atom CCID

    CCID = PNL_CCID
    if sequence(x) then
        PNL_CCID = routine_id(x)
    else
        PNL_CCID = x
    end if

    if PNL_CCID = -1 then
        PNL_CCID = CCID
        return 0
    end if

    return 1
end function

global function Read_CCID()
    return PNL_CCID
end function

------------------------------------------------------------------------------------------
-------------------------------- end of INTERFACE Section --------------------------------
------------------------------------------------------------------------------------------


------------------------------------------------------------------------------------------
---------------------------------- INTERPOLATE Section -----------------------------------
------------------------------------------------------------------------------------------
--Fastest Interpolate function. Not very smooth
function Linear_Interpolate(atom a, atom b, atom x)
    return  a*(1-x) + b*x
end function

--Medium speed Interpolate function. Fairly smooth.
function Cosine_Interpolate(atom a, atom b, atom x)
    atom f

    f = (1 - cos(x * 3.1415927)) * .5

    return a*(1-f) + b*f
end function

function Interpolate(atom a, atom b, atom x)
    if PNL_IFS then
        return Cosine_Interpolate(a, b, x)
    else
        return Linear_Interpolate(a, b, x)
    end if
end function

------------------------------------------------------------------------------------------
------------------------------ end of INTERPOLATE Section --------------------------------
------------------------------------------------------------------------------------------


------------------------------------------------------------------------------------------
---------------------------------- BASE NOISE Section ------------------------------------
------------------------------------------------------------------------------------------
global function Noise_Layer_1D(integer Gaps)
    sequence Rand
    atom Incre, total, First, Second, TilePoint
    integer posi, LenPoint, Size

    Size = PNL_SIZE_x + 3
    Rand = repeat(0, Size)
    if Gaps = 0 then
        for i = 1 to Size do
            Rand[i] = (rand(65357) - 1) * 0.000030517578125 - 1
        end for

        return Rand
    end if

    posi = 1
    Incre = 1 / (Gaps + 1)
    Second = (rand(65357) - 1) * 0.000030517578125 - 1

    while posi <= Size do
        First = Second
        Second = (rand(65357) - 1) * 0.000030517578125 - 1
        Rand[posi] = First
        posi += 1
        if posi <= Size then
            total = 0
            for j = 0 to Gaps do
                total += Incre
                if posi <= Size then
                    Rand[posi] = Interpolate(First, Second, total)
                    posi += 1
                end if
            end for
         end if
    end while

    if PNL_NTO then
        Incre = 1 / (Gaps + 1)
        TilePoint = Rand[1]
        total = Incre
        LenPoint = Size - Gaps
        for i = 0 to Gaps do
                Rand[LenPoint + i] = Interpolate(Rand[LenPoint -1], TilePoint, total)
                total += Incre
        end for
    end if

    return Rand
end function

-- Used for sound only, hence an Alt 2D or 3D is not need
global function Alt_Noise_Layer_1D(integer Gaps)
    sequence Rand
    atom Incre, total, First, Second, TilePoint
    integer posi, LenPoint, Alt, Size

    Alt = 1
    Size = PNL_SIZE_x + 3
    Rand = repeat(0, Size)
    if Gaps = 0 then
        for i = 1 to Size do
            Rand[i] = (rand(65357) - 1) * 0.0000152587890625 * Alt
            Alt *= -1
        end for

        return Rand
    end if

    posi = 1
    Incre = 1 / (Gaps + 1)
    Second = (rand(65357) - 1) * 0.0000152587890625

    while posi <= Size do
        Alt *= -1
        First = Second
        Second = (rand(65357) - 1) * 0.0000152587890625 * Alt
        Rand[posi] = First
        posi += 1
        if posi <= Size then
            total = 0
            for j = 0 to Gaps do
                total += Incre
                if posi <= Size then
                    Rand[posi] = Interpolate(First, Second, total)
                    posi += 1
                end if
            end for
         end if
    end while

    if PNL_NTO then
        Incre = 1 / (Gaps + 1)
        TilePoint = Rand[1]
        total = Incre
        LenPoint = Size - Gaps
        for i = 0 to Gaps do
                Rand[LenPoint + i] = Interpolate(Rand[LenPoint -1], TilePoint, total)
                total += Incre
        end for
    end if

    return Rand
end function

global function Noise_Layer_2D(integer Gaps)
    sequence Rand_2D, First, Second, TileRow, Empty
    atom total, Incre, LenPoint
    integer Size_x, Size_y, posi

    Size_x = PNL_SIZE_x + 3
    Size_y = PNL_SIZE_y + 3
    Rand_2D = repeat(0, Size_y)
    Empty = repeat(0, Size_x)

    if Gaps = 0 then
        for i = 1 to Size_y do
            Rand_2D[i] = Noise_Layer_1D(0)
        end for

        return Rand_2D
    end if

    posi = 1
    Incre = 1 / (Gaps + 1)
    Second = Noise_Layer_1D(Gaps)

    while posi < Size_y do
        Rand_2D[posi] = Second
        First = Second
        Second = Noise_Layer_1D(Gaps)
        posi += 1
        total = Incre
        for j = 0 to Gaps do
            if posi <= Size_y then
                Rand_2D[posi] = Empty
                for k = 1 to Size_x do
                    Rand_2D[posi][k] = Interpolate(First[k], Second[k], total)
                end for
                posi += 1
                total += Incre
            end if
        end for
    end while
    if posi <= Size_y then
        Rand_2D[posi] = Second
    end if

    if PNL_NTO then
        TileRow = Rand_2D[1]
        LenPoint = Size_y - Gaps
        total = Incre
        for i = 0 to Gaps do
            for k = 1 to Size_x do
                Rand_2D[LenPoint + i][k] = Interpolate(Rand_2D[LenPoint -1][k], TileRow[k], total)
            end for
            total += Incre
        end for
    end if

    return Rand_2D
end function

global function Noise_Layer_3D(integer Gaps)
    sequence Rand_3D, First, Second
    atom total, Incre, Size_x, Size_y, Size_z
    integer posi

    Size_x = PNL_SIZE_x + 3
    Size_y = PNL_SIZE_y + 3
    Size_z = PNL_SIZE_z + 3

    Rand_3D = repeat(0, Size_z)

    if Gaps = 0 then
        for i = 1 to Size_z do
            Rand_3D[i] = Noise_Layer_2D(0)
        end for

        return Rand_3D
    end if

    posi = 1
    Incre = 1 / (Gaps + 1)
    Second = Noise_Layer_2D(Gaps)

    while posi <= Size_z do
        First = Second
        Second = Noise_Layer_2D(Gaps)
        Rand_3D[posi] = First
        posi += 1
        if posi <= Size_z then
            total = 0
            for j = 0 to Gaps do
                total += Incre
                if posi <= Size_z then
                    Rand_3D[posi] = repeat(repeat(0, Size_x), Size_y)
                    for k = 1 to Size_y do
                        for m = 1 to Size_x do
                            Rand_3D[posi][k][m] = Interpolate(First[k][m], Second[k][m], total)
                        end for
                    end for
                    posi += 1
                end if
            end for
        end if
    end while

    return Rand_3D
end function
------------------------------------------------------------------------------------------
------------------------------- end of BASE NOISE Section --------------------------------
------------------------------------------------------------------------------------------

------------------------------------------------------------------------------------------
-------------------------------- NOISE BACKBONE Section ----------------------------------
------------------------------------------------------------------------------------------
function Noise_1D(atom x)
    return PNL_RandList[x+1]
end function

function Noise_2D(atom x, atom y)
    return PNL_RandList[x+1][y+1]
end function

function Noise_3D(atom x, atom y, atom z)
    return PNL_RandList[z+1][x+1][y+1]
end function

function SmoothedNoise_1D(atom x)
    atom Center, Sides

    Center = Noise_1D(x) * .5
    Sides  = (Noise_1D(x - 1) + Noise_1D(x + 1)) * .25
   
    return Center + Sides
end function

function SmoothedNoise_2D(atom x, atom y)
    atom corners, sides, center
    
    corners = ( Noise_2D(x-1, y-1)+Noise_2D(x+1, y-1)+Noise_2D(x-1, y+1)+Noise_2D(x+1, y+1) ) * 0.0625
    sides   = ( Noise_2D(x-1, y)  +Noise_2D(x+1, y)  +Noise_2D(x, y-1)  +Noise_2D(x, y+1) ) * 0.125
    center  =  Noise_2D(x, y) * 0.25

    return corners + sides + center
end function

function SmoothedNoise_3D(atom x, atom y, atom z)
    atom edges, corners, sides, center

    center   = Noise_3D(x, y, z) * .125

    sides    = Noise_3D(x, y, z-1)+Noise_3D(x, y, z+1)+Noise_3D(x, y-1, z)
    sides   += Noise_3D(x, y+1, z)+Noise_3D(x-1, y, z)+Noise_3D(x+1, y, z)
    sides   *= .0625

    corners  = Noise_3D(x, y-1, z-1)+Noise_3D(x, y-1, z+1)+Noise_3D(x, y+1, z-1)+Noise_3D(x, y+1, z+1)
    corners += Noise_3D(x-1, y, z-1)+Noise_3D(x-1, y, z+1)+Noise_3D(x+1, y, z-1)+Noise_3D(x+1, y, z+1)
    corners += Noise_3D(x-1, y-1, z)+Noise_3D(x-1, y+1, z)+Noise_3D(x+1, y-1, z)+Noise_3D(x+1, y+1, z)
    corners *= .03125

    edges    = Noise_3D(x-1, y-1, z-1)+Noise_3D(x-1, y-1, z+1)+Noise_3D(x-1, y+1, z-1)+Noise_3D(x-1, y+1, z+1)
    edges   += Noise_3D(x+1, y-1, z-1)+Noise_3D(x+1, y-1, z+1)+Noise_3D(x+1, y+1, z-1)+Noise_3D(x+1, y+1, z+1)
    edges   *= .015625
    
    return edges + corners + sides + center
end function

------------------------------------------------------------------------------------------
----------------------------- end of NOISE BACKBONE Section ------------------------------
------------------------------------------------------------------------------------------

------------------------------------------------------------------------------------------
-------------------------------- NOISE INTERFACE Section ---------------------------------
------------------------------------------------------------------------------------------
global function NS_Perlin_Noise_1D(atom x)
    return PNL_RandList[x+1]
end function

global function NS_Perlin_Noise_2D(atom x, atom y)
    return PNL_RandList[x+1][y+1]
end function

global function NS_Perlin_Noise_3D(atom x, atom y, atom z)
    return PNL_RandList[z+1][x+1][y+1]
end function

global function Perlin_Noise_1D(atom x)
    atom v1, v2, frac_X
    integer int_X

    int_X = floor(x)
    frac_X = x - int_X

    v1 = SmoothedNoise_1D(int_X)
    v2 = SmoothedNoise_1D(int_X + 1)

    return Interpolate(v1 , v2 , frac_X)
end function

global function Perlin_Noise_2D(atom x, atom y)
    atom v1, v2, v3, v4, i1, i2, frac_X, frac_Y
    integer int_X, int_Y

    int_X = floor(x)
    int_Y = floor(y)
    frac_X = x - int_X
    frac_Y = y - int_Y

    v1 = SmoothedNoise_2D(int_X,     int_Y)
    v2 = SmoothedNoise_2D(int_X + 1, int_Y)
    v3 = SmoothedNoise_2D(int_X,     int_Y + 1)
    v4 = SmoothedNoise_2D(int_X + 1, int_Y + 1)

    i1 = Interpolate(v1 , v2 , frac_X)
    i2 = Interpolate(v3 , v4 , frac_X)

    return Interpolate(i1 , i2 , frac_Y)
end function

global function Perlin_Noise_3D(atom x, atom y, atom z)
    sequence v, i, r
    atom frac_X, frac_Y, frac_Z
    integer int_X, int_Y, int_Z

    v = repeat(0, 8)
    i = repeat(0, 4)
    r = repeat(0, 2)

    int_X = floor(x)
    int_Y = floor(y)
    int_Z = floor(z)

    frac_X = x - int_X
    frac_Y = y - int_Y
    frac_Z = z - int_Z

    v[1] = SmoothedNoise_3D(int_X,     int_Y,     int_Z)
    v[2] = SmoothedNoise_3D(int_X + 1, int_Y,     int_Z)
    v[3] = SmoothedNoise_3D(int_X,     int_Y + 1, int_Z)
    v[4] = SmoothedNoise_3D(int_X + 1, int_Y + 1, int_Z)
    v[5] = SmoothedNoise_3D(int_X,     int_Y,     int_Z + 1)
    v[6] = SmoothedNoise_3D(int_X + 1, int_Y,     int_Z + 1)
    v[7] = SmoothedNoise_3D(int_X,     int_Y + 1, int_Z + 1)
    v[8] = SmoothedNoise_3D(int_X + 1, int_Y + 1, int_Z + 1)


    i[1] = Interpolate(v[1], v[2], frac_X)
    i[2] = Interpolate(v[3], v[4], frac_X)
    i[3] = Interpolate(v[5], v[6], frac_X)
    i[4] = Interpolate(v[7], v[8], frac_X)

    r[1] = Interpolate(i[1], i[2], frac_Y)
    r[2] = Interpolate(i[3], i[4], frac_Y)

    return Interpolate(r[1] , r[2] , frac_Z)
end function
------------------------------------------------------------------------------------------
---------------------------- end of NOISE INTERFACE Section ------------------------------
------------------------------------------------------------------------------------------
