--
-- domecalc.ex
-- v. 1.3a
--
-- post requests for help or more information on the OpenEuphoria Forum at:
-- http://openeuphoria.org/forum/index.wc
-- or email:
-- Kenneth Rhodes
-- ken_rhodes30436@yahoo.com
--
-- Computes geodesic dome "strut" lengths, given the diameter of the dome
-- and the relevant chord factors for 2, 3, and 4 Frequency, Class 1,
-- Method 1, Icosahedrons as well as 3 and 4 frequency Class 1, Method 2,
-- Icosahedrons.
--
--
-- Select from these length of measure: "cm", "m", "ft", or "in"
-- units of area and volume are keyed to the selected length of measure.
-- switch back and forth between lengths of measures. 
-- (Thanks to Jim Roberts convert.e library)
--
-- Additionally,
-- most lengths are reported in carpenters fractions as well.
-- Thanks to Derek Parnell and members of the OpenEuphoria Forum:
-- http://openeuphoria.org/forum/index.wc  

--------------------------------------------------------------------------
--------------------------------------------------------------------------
--------------------------------------------------------------------------
-- Extensive calculations for 3 Frequency, Class 1, Method 1 Icosahedrons
--------------------------------------------------------------------------
-- ** calculates the "rise" to add to L/R footwalls beneath
--    the Hexagon vertex. 
--    (Thanks to "cargoan" on the OpenEuphoria Forum)

-- ** calculates the true floor area of a dome with 5 Pent walls
--    and 5 long walls, each twice the length of a Pent wall

-- ** calculates the area gained by substituting two Hex walls for
--    a long wall.

-- ** calculates the dimensions of a triangular wedge to 
--    place on top of each Hex wall, with automatic correction
--    for a hub and strut dome or a panel dome.

-- ** calculates the length of struts offset using a correction
--    factor, if building a hub and strut dome with plywood gusset/hubs.

-- ** Default riser wall, and hub offset correction values  can be 
--    initiated in any length of measure, then if you choose to change
--    the length of measure, the program will automatically convert
--    the values to the current length of measure.
      
-- ** The user may initialize the proram by entering either the dome diameter
--    or thesphere radius. The dome diameter/radius refers to the actual foundation
--    "foot print" of the dome, specifically, the distance from the
--    center of the dome to an edge of a Pentagon wall, the dome diameter 
--    is twice that value.  
--    
--    The sphere radius is traditionally the starting point for 
--    dome calculations,  because chord factors are keyed to the 
--    sphere radius. In 3v domes, the center of the sphere is below 
--    the plane of the the bottom  Pent walls; therefore, the dome 
--    diameter probably a more "user friendly" starting point. If
--    the dome diameter is entered the program will derive the sphere
--    radius from that value. All chord/ratio calculations refer to
--    the sphere radius.
--
--    If the user prefers to enter the sphere radius, the dome diameter
--    and dome radius will be  will be derived from that value.
--    All calculations are made from the sphere radius value, (sr).
--
--    Even frequency icosahedrons truncate evenly at the hemisphere.
--    Therefore, in 2 and 4 frequency domes the dome radius/diameter
--    is equal to the sphere radius/diameter.
--------------------------------------------------------------------------
--------------------------------------------------------------------------
--------------------------------------------------------------------------


include std/console.e
include std/filesys.e
include std/mathcons.e
include std/math.e
include std/graphics.e
include std/error.e
include std/sequence.e
include std/text.e
include domecalc.e


 with type_check
 with warning
 warning_file("domecalc.err")


integer TRUE = 1, FALSE = 0
object tmp

-------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------
-- note: 2X4 inch strut = 1.5" X 3.5" (inches) or 3.81cm X 8.89cm 
--  ICOSA[13] = side "C" or "HB/HC", the hypotenuse, upward slope of left/right riser wall
--  lhvh = length of left/right riser walls between Pent riser walls.
--  HvH = rise of the end of under the Hex vertex
--  2.23 degrees = angle B, theta, angle opposite side b, triangle height
--  atom lHvH = ICOSA[13] * cos(deg2rad(2.23)) -- gives base, length "a"
--  atom HvH  = ICOSA[13] * sin(deg2rad(2.23)) -- gives height, length "b"
--              lHvH = ICOSA[13] * cos(deg2rad(2.23)) -- gives base, length "a"
--              HvH  = ICOSA[13] * sin(deg2rad(2.23)) -- gives height, length "b"
-------------------------------------------------------------------------------------------------------
-------------------------------------------------------------------------------------------------------

sequence triangles = {}
sequence lengths = {{0},{0},{0},{0},{0},{0},{0},{0},{0}}
atom a = 0, b = 0, c = 0, d = 0, e = 0, f = 0, H = 0, A = 0
 

function ht(atom a, atom b, atom c)
--  calculates height of triangle, given three sides
--  atom s =((a+b+c)*.5)  -- semi-perimeter of triangle 
--  atom A = sqrt((s*(s-a))*(s-b)*(s-c)) -- area of triangle
--  atom A = sqrt((((a+b+c)*.5)*(((a+b+c)*.5)-a))*(((a+b+c)*.5)-b)*(((a+b+c)*.5)-c)) 
--  return 2*A/b     -- h = 2*A/b  (derived from A = bh/2)
return 2*sqrt((((a+b+c)*.5)*(((a+b+c)*.5)-a))*(((a+b+c)*.5)-b)*(((a+b+c)*.5)-c))/b  
	-- derived from A = bh/2
end function

function ATabc(atom a, atom b, atom c)
    -- calculates Area of triangle, given three sides
--      atom s =(a+b+c)*.5            -- half of the triangles perimeter
--      atom A = sqrt((s*(s-a))*(s-b)*(s-c)) -- area of triangle
--      return A
return sqrt((((a+b+c)*.5)*(((a+b+c)*.5)-a))*(((a+b+c)*.5)-b)*(((a+b+c)*.5)-c))
end function

-- for 3v Class 1 Method 1:
function dome_floor_area(atom dr, atom number_Hx_walls = 0)
 atom area, a, c
    -- For 3v Class 1, Method 1 Icosahedrons:
    -- Calculate dome floor area of two triangles
    -- One triangle is formed by the base length 
    -- of a Pentagon Wall (LPW) with two legs extending
    -- to the center of the dome, essentially, the dome
    -- radius, (dr), as opposed to the sphere radius, (sr). 
    
    -- The second triangle's base is the
    -- distance from the edge of one Pentagon Wall to
    -- the next ( the distance will be twice the length
    -- of the Pentagon wall ). This length is the "long
    -- wall, which is often used to avoid building a
    -- two left/right riser walls which must have a slight
    -- elevation on one end. Together, I call these two riser walls
    -- a Hex Wall. Using two walls together (instead of a long wall)
    -- will add a little more area to the dome floor.
    
    -- The legs of the second triangle
    -- are formed in the same manner as the first.
    -- Multiplying the area of each triangle * 5 will
    -- give the area of the dome floor, with 5 long walls 
    -- twice the length of a Pent wall and 5 Pent walls,
    -- the walls supporting the base length of a Pentagon
    -- 
    
    --  simply put:
    --  The area of dome floor bounded by 5 Pentagon wall lengths
    --  and 5 longwall lengths, each = 2 X Pentagon wall length.
    --  NO Hex Extension Walls are included in these calculations!
    area = (ATabc(dr*sr, lpw, dr*sr) * 5)
    area += (ATabc(dr*sr, llw, dr*sr) * 5) 
    
    -- calculate the additional area gained by
    -- using a Hx Wall(s) (pair of llw), 
    -- instead of a long wall.
    -- ( remember this is actually two L/R walls  of length = lhvh * sr )
    if number_Hx_walls then
	b = llw*sr                -- base, length of long wall
	c = lhvh*sr               -- length of left and right legs
	area += (ATabc(c, b, c) * number_Hx_walls)
    end if
return area
end function

function hxExArea()
-- returns the area gained for each pair of Hex wall extensions
    return ATabc(lhvh*sr, llw*sr, lhvh*sr)
end function

procedure hub_offset()
    
    --  3v Domes ONLY!
    --  reduce length of struts by the hub correction factor, "hubcf".
    
    --  If constructing a WOKEY 3v (Three Frequency, Class 1 Method 1)plywood dome 
    --  with 11" diameter plywood hubs (gussets),
    --  atom hubcf = 10.8 -- cm
    
    --  If constructing a conduit dome or a pipe hub dome hubcf can be
    --  set to the required value.
    
    text_color(YELLOW)
    printf( FN, "\n\n\n\t*** HUB OFFSETS ***\n\n\tThe following lengths have been reduced by %2.2f %s\n\tto offset the width of of a plywood hub:\n", {hubcf, current_length_unit})        
    
    ICOSA[3]  = (sr * 0.40354821) - hubcf -- length "B" or "PA/HA", The base length for Pentagon & Hexagon Triangles
    ICOSA[13] = (sr * 0.41241149) - hubcf -- length "C" or "HB/HC", The side length of the Hexagon Triangles
    ICOSA[23] = (sr * 0.34861548) - hubcf -- length "A" or "PB/PC", The side length of the shorter Pentagon Triangles
    ICOSA[31] = (sr * 0.40354822) - hubcf -- length "B", again (different dihedral angle)

end procedure

procedure reset_lengths()
-- restore lengths shortened by the hub offset, "hubcf",
-- factor to original lengths
    
    ICOSA[3]  = (sr * 0.40354821) -- length "B" or "PA/HA", The base length for Pentagon & Hexagon Triangles
    ICOSA[13] = (sr * 0.41241149) -- length "C" or "HB/HC", The side length of the Hexagon Triangles
    ICOSA[23] = (sr * 0.34861548) -- length "A" or "PB/PC", The side length of the shorter Pentagon Triangles
    ICOSA[31] = (sr * 0.40354822) -- length "B", again (different dihedral angle)

end procedure


-- v = frequency 
atom dd=0, dr=0, sd=0, sr=0
public procedure set_diameter( atom x,   sequence s = "dd", integer vx = v )
    -- given x as either "dr" (the dome radius, distance from dome floor center to edge Pent/Riser wall)
    -- or:
    -- "sr" (reference sphere radius), 
    -- "sd" (reference sphere diameter), or 
    -- "dd" (dome diameter), the default,
    --  all the other parameters will be returned
    sequence ss = {"dd", "dr", "sd", "sr"}
    if not find(s, ss) then
	break
--      elsif v = 3 or v = 312 then
    elsif vx = 3 then
	if equal(sprintf("%s", {s}), "dd") then
	    dd = x
	    dr = .5 * dd
	    sd =   dd  *   1.015063688 
	    sr = .5 * sd
	elsif equal(sprintf("%s",{s}), "sd") then
	    sd = x
	    sr = x * .5
	    dd = .9851576994434 * sd 
	    dr = .5 * dd
	elsif equal(sprintf("%s", {s}), "dr") then
	    dr = x
	    dd = dr * 2
	    -- sd = sphere diameter
	    sd =   dd  *   1.015063688 
	    sr = sd * .5
	elsif equal(sprintf("%s", {s}), "sr") then
	    sr = x 
	    sd = x * 2
	    dd = .9851576994434 * sd
	    dr = .5 * dd
	end if
    else
	-- even frequency
	if equal(sprintf("%s", {s}), "dd") then
	    dd = x
	    dr = .5 * dd
	    sd =  dd
	    sr = dr
	elsif equal(sprintf("%s",{s}), "sd") then
	    sd = x
	    dd = sd 
	    sr = x * .5
	    dr = sr
	elsif equal(sprintf("%s", {s}), "dr") then
	    dr = x
	    dd = dr * 2
	    sd =   dd             -- sd = sphere diameter
	    sr = dr
	elsif equal(sprintf("%s", {s}), "sr") then
	    sr = x 
	    sd = x * 2
	    dd = sd
	    dr = sr
	end if
    end if
end procedure


sequence length_units = {"cm", "m", "in", "ft"}
sequence area_units =   {"cm2", "m2", "in2", "ft2"}
sequence volume_units = {"cm3", "m3", "in3", "ft3"}
sequence new_unit, previous_unit
sequence current_length_unit      -- default unit of measure can be changed from menu
sequence current_area_unit
sequence current_volume_unit
procedure update_units(sequence new_unit) -- new unit must be a unit of length
    
    -- update/convert sr value  to current/new unit of length measure
--      tmp = convert_unit(sr, sprintf("%s",{current_length_unit}), sprintf("%s", {new_unit}))
--      sr = tmp[3]     
--      set_diameter(sr, "sr")
    
    for x = 1 to length(length_units) do
	
	-- match volume and area units to new length unit
	
	
	if match({length_units[x]}, {new_unit}) then
	    
	current_length_unit = new_unit
	    
	    
	    -- update/convert sr value  to current/new unit of length measure
	    tmp = convert_unit(sr, sprintf("%s",{sr_unit}), sprintf("%s", {new_unit}))
	    sr = tmp[3]     
	    sr_unit = current_length_unit
	    set_diameter(sr, "sr")
	    
	    
	    -- update riser wall unit of measure
	    tmp = convert_unit(rw, sprintf("%s",{rw_unit}), sprintf("%s", {current_length_unit}))
	    rw = tmp[3]
	    rw_unit = current_length_unit
    
	    tmp = convert_unit( beam[1], sprintf("%s",{beam_unit}), sprintf("%s", {current_length_unit}))
	    beam[1] = tmp[3]
	    tmp = convert_unit( beam[2], sprintf("%s",{beam_unit}), sprintf("%s", {current_length_unit}))
	    beam[2] = tmp[3]
	    beam_unit = current_length_unit
	    
	    -- update hub correction factor unit of measure
	    tmp = convert_unit( hubcf, sprintf("%s", {hub_unit}), sprintf("%s",{current_length_unit}))
	    hubcf = tmp[3]
	    
	    tmp = convert_unit( hubcf_default, sprintf("%s", {hub_unit}), sprintf("%s",{current_length_unit}))
	    hubcf_default = tmp[3]
	    hub_unit = current_length_unit
	    
	    -- update wedgecf unit of measure
	    tmp = convert_unit( wedgecf, sprintf("%s", {wedgecf_unit}), sprintf("%s",{current_length_unit}))
	    wedgecf = tmp[3]
	    wedgecf_unit = current_length_unit
	    
	    current_area_unit = sprintf("%s",{area_units[x]})
	    
	    -- update area unit of measure for a sheet of plywood
	    tmp = convert_unit( aps, sprintf("%s", {aps_unit}), sprintf("%s",{current_area_unit}))
	    aps = tmp[3]
	    aps_unit = current_area_unit
	    
	    current_volume_unit = sprintf("%s", {volume_units[x]})
	    
	end if
	
    end for
    
end procedure

---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
--
-- Riser Wall Height and unit of measure:
--
-- If you change the 'current_length_unit' of measure above, it is not necessary
-- to change the riser wall unit 'rw_unit' below to match the 'current-unit'.

-- The 'rw_unit' measure will always be expressed in current_length_units.
-- i.e. if you change the current unit of measure  to m, cm, or in  and leave
-- the riser wall default @ 121.920cm,  The riser wall will be reported as 1.219m, 121.920cm, or 48in,
-- according to the current unit of measure:

-- Likewise, if you change the rw_unit default unit of measure from "ft" to another new unit
-- of measure, say 'cm' the value for 'rw' is now initiated at program start in centimeters,
-- but will be always be converted to the current unit of measure. I have decided to
-- initiate the riser wall unit of measure using inches. But this can be changed to 
-- the user's peference.

sequence rw_unit  --  "in", "ft", "cm", or "m" 
atom rw           -- designate in rw_units

--  sequence rw_unit = "in"
--  atom rw = 54   -- default riser wall height is set to 54 inches

rw_unit = "ft"
 rw = 4.5   -- default riser wall height is set to 4.5 ft = 54 inches
--  rw = 4.0

--  rw_unit = "cm"
-- rw = 51.9     -- cm  WOKEY 32', 1'8-7/16

--  rw_unit = "cm"
--  rw = 117.2      -- cm WOKEY 20', 3'10-1/8 inches
-- rw = 137.16 cm      -- 4.5ft
-- rw = 121.920 cm     -- most 3v domes have a riser wall: '121.920cm = 4ft = 48in',

---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
--
--  default unit
-- for area of a 4x8 ft plywood sheet
--
sequence aps_unit = "ft2"
atom  aps = 32                    -- 32 square ft ("ft2") 
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
-- 
-- hub/gusset correction factor = 
-- default value 10.8 cm ( offset of WOKEY plywood hub)
--
atom hubcf
sequence hub_unit = "cm"
atom hubcf_default = 10.8 -- cm
hubcf = hubcf_default
integer HUB  --  FALSE  -- TRUE, if you wish to start program with hub correction factor 
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
-- beam dimensions   width X depth
-- use to compute actual interior floor space
sequence beam_unit = "in"
object beam = {1.5,3.5}  -- for 2 X 4 standard lumber
--  object beam = {1.5,5.5}  -- for 2 X 6 standard lumber
--
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
--  3v wedge offset factor for L/R Hex walls
--  for hub and strut domes, not required by panel domes   
atom wedgecf
sequence wedgecf_unit = "in"
wedgecf = 1.5 * .5  -- half breadth of 2" X 4"
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
--
--  These are for 3 Frequency Class 1, Method 1 domes ONLY:
atom PHA,                         --  Pentagon base ( "a" length of triangle pabc )
HTpabc,                           -- height of Pabc
PBC,
HA, 
HThabc,                           -- height of habc
HBC,
--
cpw,                              --  1
clw,                              --  2
lpw,                              --  3
llw,                              --  4
lvhxb,                            --  5
lvptapx,                          --  6
lvhxvx,                           --  7
lvtptb,                           --  8
lvtptapx,                         --  9
hvh,                              --  10
lhvh ,                            --  11
drhvh                             --  12
--  ddr                               --  13
--------------------------------------------
-- sphere radius (sr) chord factors/ratios:
PHA = 0.403548212237891           -- pent base
PBC = 0.348615491011437           -- pent legs
HTpabc = 0.2842885342             -- height of Pentagon triangle, pabc
HA  = 0.403548212237891           -- Hex base
HBC = 0.412411489396122           -- Hex legs
HThabc = 0.3596810346             -- height of each Hex triangle, habc
cpw = 0.964275457758272           --  1
clw = 0.898715076289873           --  2
lpw = 0.403548214208210           --  3
llw = 0.807096424475781           --  4
lvhxb = 0.343278613075557         --  5
lvptapx = 0.555436463608380       --  6
lvhxvx  = 0.623015164936134       --  7
lvtptb   = 0.767594313550107      --  8
lvtptapx = 0.828360693363705      --  9
hvh     = 0.016048254107185       --  10  -- rise, of left/right walls under Hex vertex
lhvh    = 0.412100250403204       --  11
drhvh   = 0.982237875107596       --  12
--  ddr     = 0.9851576994434         --  13

-- default data is for 3vC1M1 domes
object ddata = {
dd, sd, 
dr, sr, 
dr, 
cpw, clw, drhvh, 
lpw, llw, lhvh,
rw, hvh,
lvhxb, 
lvptapx,lvhxvx,
lvtptb,lvtptapx,
--   hvh,lhvh,drhvh, ddr,
PHA,HTpabc,PBC,
HA, HThabc,HBC}

-- These are for 3v class 1, method 1 Icosahedron:
sequence ddata_labels = {
"\n\n\tDome Diameter:", 
"\tSphere Diameter",
"\n\tDome Radius:", 
"\t\tSphere Radius:",
"\n\n\tDistance dome floor center to:" &
"\n\t\tedge, Pent Wall:", 
"\n\t\tcenter, Pent Wall:", 
"\tcenter, Long Wall:", 
"\n\t\tcenter edge between L/R riser walls, beneath Hex Vertex:",

"\n\n\tLength:\n\t\tPent Base/Short Wall:",
"\tLong Wall:",

"\n\t\tLength Hex Base Extension Wall:",

"\n\n\tElevations:\n\t\tRiser Wall:", 
"\t\t\tRise of Hex Vertex:",

"\n\t\tBottom Hex Base:",
"\t\tApex Bottom Pent:",

"\n\t\tHex Vertex:",
"\t\t\tTop Pent Base:",
"\n\t\tApex:"
}
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------

procedure show_unit_of_measure()  -- all units of measure are based on the current_length_unit
    
    text_color(CYAN)
    puts(1, "Change current unit of measure from ")
    text_color(BROWN)
    printf(1,"%s ", {current_length_unit})
    text_color(CYAN)
    display("to:",0)
    
    for x = 1 to length(length_units) do
	if not equal({current_length_unit}, {length_units[x]}) then
	    text_color(BROWN)
	    printf(1," %s", {length_units[x]})
	end if
    end for
    
    text_color(CYAN)
    display(": ",0)
    text_color(RED)
    
end procedure


-----------------------------------------------------------------------------
---------------- start program:
constant TYPE = 1
object ICOSA = {}
object TwoFrqIcosa = {}, ThreeFrqIcosa = {}, FourFrqIcosa = {},
--  ThreeFrqIcosaClassOneMethodTwo = {},
FourFrqIcosaClassOneMethodTwo = {}

integer v = 3          -- default frequency

TwoFrqIcosa =  {                  -- data from DomeBook 2
"2v (Two Frequency), Class 1, Method 1, Icosahedron:",
"Length A    1,1 1,0    ",   0.61803399,
"Axial       0,0 1,0 1,1",  71.999996,
"Face        1,1 0,0 1,0",  58.861974,
"Face        1,1 2,1 1,0",  59.999098,
"Dihedral    1,1 1,0    ", 161.970892,

"Length B    2,1 2,0    ",   0.54653306,
"Axial       0,0 2,0 2,1",  74.141260,
"Face        2,1 1,0 2,0",  55.569010,
"Dihedral    2,1 2,0    ", 157.571075
}


ThreeFrqIcosa = {                 -- Chord Factors from DomeBook 2:

"3v (Three Frequency) , Class 1, Method 1, Icosahedron:",
--------------------------------------------------------------------------------------
"Length B    1,1 1,0    ", 0.40354821, -- *Base Length of Pentagon/Hexagon Triangles
"Axial       0,0 1,0 1,1",  78.359272,
"Face        1,1 0,0 1,0",  70.730537,
"Face        1,1 2,1 1,0",  58.583164,
"Dihedral    1,1 1,0    ", 168.641064,
--------------------------------------------------------------------------------------
"Length C    2,1 2,0    ",   0.41241149,
"Axial       0,0 2,0 2,1",  78.099906,
"Face        2,1 1,0 2,0",  60.708416,
"Face        2,1 3,1 2,0",  60.708416,
"Dihedral    2,1 2,0    ", 166.421442,
--------------------------------------------------------------------------------------
"Length A    3,1 3,0    ",   0.34861548,
"Axial       0,0 3,0 3,1",  79.961621,
"Face        3,1 2,0 3,0",  54.634727,
"Dihedral    3,1 3,0    ", 165.564739,
--------------------------------------------------------------------------------------
"Length B    3,2 3,1    ", 0.40354822, -- *Base Length of Hexagon/Pentagon Triangles
"Axial       0,0 3,1 3,2",  78.359272,
"Face        3,2 2,1 3,1",  58.583164,
"Dihedral    3,2 3,1    ", 165.542280
}


-- IMPORTANT: Dome Diameter calculations for the 
-- 3v (Frequency) Class 1, Method 2 Icosahedrons are not accurate,
-- Howevever, the strut/edge lengths should be accurately calculated
-- from the sphere radius.
-- 

--  ThreeFrqIcosaClassOneMethodTwo = { -- data from Domebook 2
--  "3v (Frequency) , Class 1, Method 2, Icosahedron:",
--  --------------------------------------------------------------------------------------
--  "Length B    1,1 1,0    ", 0.4240625, -- *Base Length of Pentagon/Hexagon Triangles
--  "Axial       0,0 1,0 1,1",  77.75857,
--  "Face        1,1 0,0 1,0",  70.59285,
--  "Face        1,1 2,1 1,0",  63.27959,
--  "Dihedral    1,1 1,0    ", 169.34601,
--  --------------------------------------------------------------------------------------
--  "Length C    2,1 2,0    ",   0.4041944,
--  "Axial       0,0 2,0 2,1",  78.34037,
--  "Face        2,1 1,0 2,0",  63.00334,
--  "Face        2,1 3,1 2,0",  58.36019,
--  "Dihedral    2,1 2,0    ", 166.66061,
--  --------------------------------------------------------------------------------------
--  "Length A    3,1 3,0    ",   0.3669588,
--  "Axial       0,0 3,0 3,1",  79.42750,
--  "Face        3,1 2,0 3,0",  54.70356,
--  "Dihedral    3,1 3,0    ", 164.81383,
--  --------------------------------------------------------------------------------------
--  "Length A    3,2 3,1    ", 0.3669588, -- *Base Length of Hexagon/Pentagon Triangles
--  "Axial       0,0 3,1 3,2",  79.42150,
--  "Face        3,2 2,1 3,1",  53.99331,
--  "Dihedral    3,2 3,1    ", 164.49862
--  }
--  
 
FourFrqIcosa = {                  -- data from Domebook 2
"4v (Four Frequency), Class 1, Method 1, Icosahedron:", -- 1

"Length B    1,1 1,0    ",        --2
0.29524181,                       --3

"Axial       0,0 1,0 1,1",        --4
81.510921,                        --5

"Face        1,1 0,0 1,0",        --6
71.331604,                        --7

"Face        1,1 2,1 1,0",        --8
60.159766,                        --9

"Dihedral    1,1 1,0    ",        --10
172.197790,                       --11

"Length D    2,1 2,0    ",        --12
0.31286893,                       --13

"Axial       0,0 2,0 2,1",        --14
80.999996,                        --15

"Face        2,1 1,0 2,0",        --16
63.668768,                        --17

"Face        2,1 3,1 2,0",        --18
58.717473,                        --19

"Dihedral    2,1 2,0    ",        --20
169.981901,                       --21

"Length C    3,1 3,0    ",        --22
0.29453084,                       --23

"Axial       0,0 3,0 3,1",        --24
81.531510,                        --25

"Face        3,1 2,0 3,0",        --26
57.534353,                        --27

"Face        3,1 4,1 3,0",        --28
59.920114,                        --29

"Dihedral    3,1 3,0    ",        --30
169.617161,                       --31

"Length E    3,2 3,1    ",        --32
0.32491969,                       --33

"Axial       0,0 3,1 3,2",        --34
80.650292,                        --35

"Face        3,2 2,1 3,1",        --36
59.999999,                        --37

"Face        3,2 4,2 3,1",        --38
62.565048,                        --39

"Dihedral    3,2 3,1    ",        --40
169.642082,                       --41

"Length A    4,1 4,0    ",        --42
0.25318459,                       --43

"Axial       0,0 4,0 4,1",        --44
82.727277,                        --45

"Face        4,1 3,0 4,0",        --46
54.334194,                        --47

"Dihedral    4,1 4,0    ",        --48
169.490046,                       --49

"Length F    4,2 4,1    ",        --50
0.29858813,                       --51

"Axial       0,0 4,1 4,2",        --52
81.413979,

"Face        4,2 3,1 4,1",        --53
58.796873,                        --54

"Dihedral    4,2 4,1    ",        --55
169.505737                        --56
}

FourFrqIcosaClassOneMethodTwo = { -- data from Domebook 2
"4v (Four Frequency), Class 1, Method 2, Icosahedron:", -- 1

"Length E    1,1 1,0    ",        --2
0.321244,                         --3

"Axial       0,0 1,0 1,1",        --4
80.75699,                         --5

"Face        1,1 0,0 1,0",        --6
71.20597,                         --7

"Face        1,1 2,1 1,0",        --8
65.29889,                         --9

"Dihedral    1,1 1,0    ",        --10
172.5324,                         --11

"Length C    2,1 2,0    ",        --12
0.3132066,                        --13

"Axial       0,0 2,0 2,1",        --14
80.99019,                         --15

"Face        2,1 1,0 2,0",        --16
66.05975,                         --17

"Face        2,1 3,1 2,0",        --18
61.43574,                         --19

"Dihedral    2,1 2,0    ",        --20
171.0111,                         --21

"Length D    3,1 3,0    ",        --22
0.2977251,                        --23

"Axial       0,0 3,0 3,1",        --24
81.43897,                         --25

"Face        3,1 2,0 3,0",        --26
60.31862,                         --27

"Face        3,1 4,1 3,0",        --28
57.35055,                         --29

"Dihedral    3,1 3,0    ",        --30
169.5426,                         --31

"Length B    3,2 3,1    ",        --32
0.2995157,                        --33

"Axial       0,0 3,1 3,2",        --34
81.38710,                         --35

"Face        3,2 2,1 3,1",        --36
60.00000,                         --37

"Face        3,2 4,2 3,1",        --38
57.12850,                         --39

"Dihedral    3,2 3,1    ",        --40
169.3659,                         --41

"Length A    4,2 4,1    ",        --42
0.2759044,                        --43

"Axial       0,0 4,0 4,1",        --44
82.07062,                         --45

"Face        4,1 3,0 4,0",        --46
54.39701,                         --47

"Dihedral    4,1 4,0    ",        --48
168.5529                          --49
}


---------------------------------------------------------------------------------------------------
---------------------------------------------------------------------------------------------------
--
-- start program
--
------------------------------------------------
------------Default Values----------------------
--
-- set default Icosahedron Frequency
--
--  integer v                         -- frequency

--  ICOSA = ThreeFrqIcosa  -- Class 1, Method 1, Icosahedron
--  v = 3
--
--  ICOSA = FourFrqIcosaClassOneMethodTwo
--  integer v = 412  -- Class 1, Method 1, Icosahedron
--  
 ICOSA = TwoFrqIcosa -- Class 1, Method 1, Icosahedron
 v = 2
--
--  ICOSA = FourFrqIcosa  -- Class 1, Method 1, Icosahedron
--  v = 4
--
------------------------------------------------
------------------------------------------------
integer carpenters_frac = FALSE   -- default can be toggled from menu
------------------------------------------------
------------------------------------------------
--
--  set default values for program at start up
--
--  sr = 495.1*2 -- cm                            
--  tmp  = convert_unit(sr, "cm", "ft")  -- convert to cm
--  sr = tmp[3]

--  dd = 39                           -- ft
--   tmp  = convert_unit(dd, "ft", "cm") -- convert to cm
--   dd = tmp[3]
--  set_diameter(dd, "dd")

--  sr = 295.8 * 2 -- cm for WOKEY ~40ft plywood hub dome
--   tmp  = convert_unit(dd, "ft", "cm") -- convert to cm
--   dd = tmp[3]

sequence sr_unit = "cm"
--  sr = 485.1 -- cm WOKEY ~32ft
--  sr = 295.8 * 2 -- cm WOKEY ~40 FT
--  sr_unit = "in"
--  sr = (9*12)+ 8.5 -- in
--  set_diameter(sr, "sr")
--  dd = 39 -- ft
--  sr_unit = "ft"

-- SCALE 24.900 ft dd to in = 758.952cm / 12 
--  dd = 758.952  -- cm 

-- for 2v model 25.890ft dd to in  / 12  
--  dd = 25.890/24 --  

-- for 18 in model
dd = 18 -- in
    tmp  = convert_unit(dd, "in", "cm") -- convert to cm
    dd = tmp[3]
set_diameter(dd, "dd")
------------------------------------------------
------------------------------------------------
-- 3v Class 1 Method 1 Domes only
-- a little more area can be inclosed
-- by using two short riser walls instead
-- of a long wall between two Pentagon Riser Walls
atom hxExtensions = 0 
------------------------------------------------
------------------------------------------------
HUB = FALSE
------------------------------------------------
------------------------------------------------

-- initial units of measure for 
-- length, area, and volume:
current_length_unit = "cm"
current_area_unit = "cm2"
current_volume_unit = "cm3"

-- all units of measure are easily "updated"   
-- to a new or current length_unit as follows:
--  update_units("ft")                -- synchronizes units to "ft", "ft2", "ft3"
update_units("cm")
------------------------------------------------
------------------------------------------------

procedure set_lengths()
    
    if v = 2 then
	
	ICOSA = TwoFrqIcosa
	
	ICOSA[3]    =   sr * ICOSA[3]
	ICOSA[13]   =   sr * ICOSA[13]
	
	triangles = {"AAA", "BAB"}
	a = ICOSA[3]    b = ICOSA[13]
	H = ht(a,a,a)   A = ATabc(a,a,a) lengths[1] = {A,H,a,a,a}
	H = ht(b,a,b)   A = ATabc(b,a,b) lengths[2] = {A,H,b,a,b}
	
    elsif v = 3 then
	
	
	ICOSA = ThreeFrqIcosa
	
	-- convert chord factors to lengths (chords)
	ICOSA[3]  *= sr           -- length "B" or "PA/HA", The base length for Pentagon & Hexagon Triangles
	ICOSA[13] *= sr           -- length "C" or "HB/HC", The side length of the Hexagon Triangles
	ICOSA[23] *= sr           -- length "A" or "PB/PC", The side length of the shorter Pentagon Triangles
	ICOSA[31] *= sr           -- length "B", again.
	
	ddata *= sr
	
	
	triangles = {"ABA", "CBC", "wHvH"}
	b = ICOSA[3]    c = ICOSA[13]      a = ICOSA[23]
	H = ht(b,a,a)   A = ATabc(a,b,a)   lengths[1] = {A,H,a,b,a}
	H = ht(b,c,c)   A = ATabc(c,b,c)   lengths[2] = {A,H,c,b,c}
	
	-- wHvH--------------------------------------------------------------
	-- wedge to place on top L/R wall under Hexagon vertex
	--c                 -- side "C" or "HC", leg lengths of Hexagon triangles  
	b = lhvh * sr       -- base or "run" of L/R Hex wall triangle 
	H = (hvh * sr)      -- height or rise of L/R Hex wall triangle
	if HUB then
	    
	    H = 0.009608329952671 * sr 
	    b = 0.2467438505747   * sr
	    c = 0.246923597025    * sr
	
	-- offset: half the breadth of 2" X 4" for Hub and strut domes
	-- because the edges of plywood triangular panel skins meet 
	-- in the middle of the strut. The wedge will fit below the strut.
	-- note: Panel domes do not need this wedge correction factor as
	-- the plywood triangular panels extend all the way to the edge of
	-- of the panel side.
	
	end if
	
	-- area, just to "fit" in sequence "lengths[3]"
	-- right triangle
	A = (b * H) * .5                                          
	lengths[3] = {A, H,c,b,H} 
	-- WedgeHvH--------------------------------------------------------------
	
	
--      elsif v = 312 then
--          
--          ICOSA = ThreeFrqIcosaClassOneMethodTwo
--          
--          -- convert chord factors to lengths (chords)
--          ICOSA[3]  *= sr           -- length "B"
--          ICOSA[13] *= sr           -- length "C" 
--          ICOSA[23] *= sr           -- length "A" 
--          ICOSA[31] *= sr           -- length "A", again.
--          
--          ddata *= sr
--          
--          triangles = {"ABA", "CAC", "CBC"}
--          
--          b = ICOSA[3]    c = ICOSA[13]      a = ICOSA[23]
--          
--          H = ht(a,b,a)   A = ATabc(a,b,a)   lengths[1] = {A,H,a,b,a}
--          H = ht(c,a,c)   A = ATabc(c,a,c)   lengths[2] = {A,H,c,a,c}
--          H = ht(c,b,c)   A = ATabc(c,b,c)   lengths[3] = {A,H,c,b,c}
	
	
    elsif v = 4 then
	
	ICOSA = FourFrqIcosa
	
	ICOSA[3]  = sr * ICOSA[3]
	ICOSA[13] = sr * ICOSA[13]
	ICOSA[23] = sr * ICOSA[23]
	ICOSA[33] = sr * ICOSA[33]
	ICOSA[43] = sr * ICOSA[43]
	ICOSA[51] = sr * ICOSA[51]
	
	triangles = {"ABA", "CBC", "CFD", "DFC", "DED", "EEE"}
	b = ICOSA[3]    d = ICOSA[13]   c = ICOSA[23]   
	e = ICOSA[33]   a = ICOSA[43]   f = ICOSA[51]
	
	
	H = ht(a,b,a)   A = ATabc(a,b,a)   lengths[1] = {A,H,a,b,a}
	H = ht(c,b,c)   A = ATabc(c,b,c)   lengths[2] = {A,H,c,b,c}
	H = ht(c,f,d)   A = ATabc(c,f,d)   lengths[3] = {A,H,c,f,d}
	H = ht(d,f,c)   A = ATabc(d,f,c)   lengths[4] = {A,H,d,f,c}
	H = ht(d,e,d)   A = ATabc(d,e,d)   lengths[5] = {A,H,d,e,d}
	H = ht(e,e,e)   A = ATabc(e,e,e)   lengths[6] = {A,H,e,e,e}
	
	
    elsif v = 412 then
	-- 4v Class 1, Method 2, Icosahedron
	
	ICOSA = FourFrqIcosaClassOneMethodTwo
	
	ICOSA[3]  *= sr     -- side "E"
	ICOSA[13] *= sr     -- side "C"
	ICOSA[23] *= sr     -- side "D"
	ICOSA[33] *= sr     -- side "B"
	ICOSA[43] *= sr     -- side "A"
	
	triangles = {"AEA", "DAC", "CAD", "DED", "CBC", "BBB"}
	e = ICOSA[3]    c = ICOSA[13]   d = ICOSA[23]   
	b = ICOSA[33]   a = ICOSA[43]  
		
	H = ht(a,e,a)   A = ATabc(a,e,a)   lengths[1] = {A,H,a,e,a}
	H = ht(d,a,c)   A = ATabc(d,a,c)   lengths[2] = {A,H,d,a,c}
	H = ht(c,a,d)   A = ATabc(c,a,d)   lengths[3] = {A,H,c,a,d}
	H = ht(d,e,d)   A = ATabc(d,e,d)   lengths[4] = {A,H,d,e,d}
	H = ht(c,b,c)   A = ATabc(c,b,c)   lengths[5] = {A,H,c,b,c}
	H = ht(b,b,b)   A = ATabc(b,b,b)   lengths[6] = {A,H,b,b,b}
	
    end if
    
end procedure


procedure menu()
    
    -- show essential data for current dome above menu:
    text_color(YELLOW)   printf(SCREEN, "\n\n\t%s ", {ICOSA[TYPE]})
    text_color(CYAN)    puts(SCREEN, "\n\n\tCurrent Units of Measure: ")
    
    text_color(BROWN)   
    printf(SCREEN, "%s %s %s\n", {current_length_unit, current_area_unit,current_volume_unit})
    
    text_color(CYAN)    puts(SCREEN, "\n\tDome Diameter  / Sphere Diameter  / Sphere Radius")
    
    text_color(RED)     printf(SCREEN, "\n\t%6.3f ", dd)
    text_color(BROWN)   printf(SCREEN, "%s", {current_length_unit})
    
    text_color(RED)     printf(SCREEN, "\t\t\t%6.3f ", sd)
    text_color(BROWN)   printf(SCREEN, "%s", {current_length_unit})
    
    text_color(RED)     printf(SCREEN, "\t\t\t%6.3f ", sr)
    text_color(BROWN)   printf(SCREEN, "%s\n", {current_length_unit})
    
    
    
    sequence options = {
    "Enter/change diameter of dome, expressed in ", -- 1
    "Calculate data from radius of reference sphere - enter sphere radius in ", -- 2
    "Enter/change height of riser wall, expressed in ", -- 3
    " ", -- 4  change current unit of measure / current_length_unit
    "Toggle Carpenter\'s Fractions on/off", -- 5
    "Select/Change Frequency of Icosahedron -{2, 3, 4, 412}: ", -- 6
    "Select or enter value for hub offset", -- 7
    "Display data for " & trim(sprintf("%6.2f%s%dvDome", {dd,current_length_unit,v})), -- 8
    "Write Current Data to file: " & trim(sprintf("%6.0f%s%dvDome.txt", {dd,current_length_unit,v})), -- 9
    "Quit - or enter \'0\'\n\n" -- 10
    }
    
    if v = 3 then
	
	-- add an option to enter # of Hex extensions, if desired. Default is set to 0
	options = insert(options, "Enter/edit number of Hex wall extensions", 10)
    
    else
	
	if length(options) = 11 then
	options = options[1..10-1] & options[10+1..length(options)]
	end if
    
    end if
    
    
    -- display menu options
    for x = 1 to length(options) do
	text_color(YELLOW)
	if x < 10 then
	    printf(SCREEN, "\n\t %d: ", x)
	else
	    printf(SCREEN, "\n\t%d: ", x)
	end if
	text_color(CYAN)
	if x < 4 then
	    printf(SCREEN, "%s", {options[x]})
	    text_color(BROWN)
	    printf(SCREEN, "%s ", {current_length_unit})
	elsif x = 4 then
	    show_unit_of_measure()
	else
	    text_color(CYAN)
	    printf(SCREEN, "%s", {options[x]})
	end if
    end for
    
    
    
    -- select menu options
    text_color(YELLOW)
    
    integer menu_item = prompt_number("\t#  ", {0,length(options)})
    if menu_item = 1 then
	-- compute data from dome (floor) diameter
	
	printf(SCREEN, "\t%s", {sprintf("%s",{options[1]})})
	text_color(BROWN)
	printf(SCREEN, "%s" , {current_length_unit})
	text_color(YELLOW)
	dd = prompt_number("\t#:  ", {1, 2000})
	set_diameter(dd , "dd")
	
    
    elsif menu_item = 2 then
	-- compute data from radius of reference sphere
	
	printf(SCREEN, "\t%s", {sprintf("%s",{options[2]})})
	text_color(BROWN)
	printf(SCREEN, "%s" , {current_length_unit})
	text_color(YELLOW)
	printf(SCREEN, "\t#: ")
	sr = prompt_number(" ", {1,2500})
	set_diameter(sr, "sr")
	
    
    elsif menu_item = 3 then
	-- change height of riser wall (expressed in current_length_unit)
	
	printf(SCREEN, "\t%s", {sprintf("%s",{options[3]})})
	text_color(BROWN)
	printf(SCREEN, "%s" , {current_length_unit})
	text_color(YELLOW)
	printf(SCREEN, "\t#: ")
	rw = prompt_number(" ", {2, 1000})
	
    
    elsif menu_item = 4 then
	-- change unit of measure (current_length_unit)
	
	new_unit = " "
	loop  do
	    show_unit_of_measure()
	    new_unit = prompt_string("new unit: ")
	    until find(new_unit, length_units)
	end loop
	
	update_units(new_unit)
	
    
    elsif menu_item = 5 then
	-- toggle carpenters fractions on/off
	
	if carpenters_frac = FALSE then
	    carpenters_frac = TRUE
	else
	    carpenters_frac = FALSE
	end if
	
    
    elsif menu_item = 6 then
	--          -- change frequency of Icosahedron

	printf(SCREEN, "\t%s", {sprintf("%s",{options[6]})})
	puts(SCREEN, "\n\t412 = 4v (Four Frequency), Class 1, Method 2, Icosahedron")
	text_color(YELLOW)
	printf(SCREEN, "\n\t\t#: ")
	tmp = {2,3,4,412}
--          tmp = {2,3,4,312,412}
	loop  do
	    v = prompt_number("", {2,412})
	    until find(v, tmp)
	end loop
	set_diameter(dd, "dd", v)
	tmp = {}
	
    
    elsif menu_item = 7 then
	
	text_color(CYAN)
	printf(SCREEN, "\t%s", {sprintf("%s",{options[6]})})
	
	text_color(CYAN)
	puts(SCREEN, "\n\n\t0 = Do not use of hub offset factor")
	
	puts(SCREEN, "\n\n\t1 = Default: use ")
	text_color(YELLOW)
	printf(SCREEN, "%2.2f%s ", {hubcf_default, current_length_unit})
	
	text_color(CYAN)
	puts(SCREEN, "\n\n\t2 = Enter new value for hub offset,\n\t\texpressed in ")
	text_color(YELLOW)
	printf(SCREEN, "%s", {current_length_unit})
	
	hubcf = prompt_number("\n\n\t\t\t#: ", {0,2})
	
	if hubcf = 0 then
	    hubcf = hubcf_default
	    HUB = FALSE
	elsif hubcf = 1 then
	    hubcf = hubcf_default
	    HUB = TRUE
	elsif hubcf = 2 then
	    hubcf = hubcf
	    HUB = TRUE
	end if
	
	text_color(YELLOW)
    
    elsif menu_item = 8 then
	-- do nothing, current dome diameter data will be displayed
	
    elsif menu_item = 9 then
	-- write current data to file
	
	out = sprintf("%2.0f%s%dvDome.txt", {dd,current_length_unit,v})
	FN = open(out, "w")
	hardcopy = TRUE
	
    elsif v = 3 and menu_item = 10 then
	
	printf(SCREEN, "\nEach Hex wall extension = %3.2f %s", {hxExArea(), current_area_unit})
	hxExtensions = prompt_number("\nEnter number of paired Hex wall extensions (0-5): ", {0,5})
    
    elsif menu_item = length(options) or menu_item = 0 then
	-- quit program
	
	bk_color(WHITE)
	text_color(BLACK)
	clear_screen()
	free_console()
	abort(1)
	
    end if
    
    clear_screen()  
    
end procedure


procedure report_strut_lengths()
    -- report strut or edge lengths
    integer side, axial, dihedral
    side = TRUE
    axial = FALSE
    dihedral = FALSE
    
    for x = 2 to length(ICOSA) do
	
	if sequence(ICOSA[x]) then
	    
	    if match("Length", ICOSA[x]) then
		
		text_color(BROWN)
		printf(FN,"\n\t%s\t",  {ICOSA[x]})
		axial = FALSE
		dihedral = FALSE
		side = TRUE
		
	    elsif match("Axial", ICOSA[x])  then
		
		text_color(CYAN)
		printf(FN,"\t%s\t",  {ICOSA[x]})
		side = FALSE
		dihedral = FALSE
		axial = TRUE
		
	    elsif match("Dihedral", ICOSA[x]) then
		
		text_color(CYAN)
		printf(FN,"\t%s\t",  {ICOSA[x]})
		side = FALSE
		axial = FALSE
		dihedral = TRUE
		
	    else
		text_color(CYAN)
		printf(FN,"\t%s\t",  {ICOSA[x]})
		side = FALSE
		axial = FALSE
		dihedral = FALSE
		
	    end if
	    
	elsif atom(ICOSA[x]) then
	    
	    if side then
		
		text_color(CYAN)
		printf(FN,"\t%3.3f", {ICOSA[x]})

		text_color(BROWN)
		printf(FN, " %s", {current_length_unit})
		
		text_color(YELLOW)
		puts(FN," = ")
		
		if not match(current_length_unit, "m") then
		    
		    previous_unit = current_length_unit -- so we can convert back
		    new_unit = convert_unit(ICOSA[x], current_length_unit, "m")
		    ICOSA[x] = new_unit[3] -- now expressed in meters
		    
		    -- convert meters to carpenters inch fractions
		    text_color(CYAN)
		    printf(FN,"%d %d/%d inches\n", carp(ICOSA[x], m_in, 32))
		    
		    -- convert back to "current_length_unit"
		    current_length_unit = previous_unit
		    new_unit = convert_unit(ICOSA[x], "m", current_length_unit)
		    ICOSA[x] = new_unit[3]
		    
		else
		    --                  current_length_unit is "m"
		    printf(FN,"%d %d/%d inches\n", carp(ICOSA[x], m_in, 32))
--                      use following line instead if you prefer resolution of 1/16th inch
--                      printf(FN,"%d %d/%d inches\n", carp(ICOSA[x], m_in, 16))
		end if
		
		side = FALSE
		axial = TRUE
		
	    elsif axial then
		
		text_color(GREEN)
		printf(FN,"\t%3.3f/%3.3f\n", {ICOSA[x],90-ICOSA[x]})
		side = FALSE
		axial = FALSE
		
	    elsif dihedral then
		
		text_color(RED)
		 printf(FN,"\t%3.3f\t\tbevel = %3.2f \n", {ICOSA[x], ((ICOSA[x]-180)*.5)})
--                   printf(FN,"\t%3.3f\t\tbevel = %3.2f \n", {ICOSA[x], (ICOSA[x]-180)})
		dihedral = FALSE
		axial = FALSE
		side = TRUE
		
	    else
		
		text_color(GREEN)
		printf(FN,"\t%3.3f\n", {ICOSA[x]})
		dihedral = FALSE
		axial = FALSE
		side = FALSE
		
	    end if
	    
	end if
	
    end for
    
end procedure



procedure report_dome_data()
    
    clear_screen()
    text_color(YELLOW)  printf(FN, "\t%s", {ICOSA[TYPE]} )
    text_color(BROWN)    puts(FN, "\n\n\tCurrent Units of Measure: ")
    text_color(CYAN)     printf(FN, "%s %s %s", {current_length_unit, current_area_unit, current_volume_unit})
    
    
    
    if v = 3 then
	 
	ddata =   {
	dd, sd,
	dr, sr,
	dr,  -- dome radius, from dome center to edge of Pentagon wall
	--
	cpw*sr, clw*sr, drhvh*sr,
	lpw*sr, llw*sr, lhvh*sr,
	
--          rw, 
	rw - (hvh*sr),   
	hvh*sr,
	
	lvhxb       *   (sr  +   rw),
	lvptapx     *   (sr  +   rw), 
	lvhxvx      *   (sr  +   rw),
	lvtptb      *   (sr  +   rw), 
	lvtptapx    *   (sr  +   rw)
	
	}
	
    else
	
	ddata =   {
	dd, sd,
	dr, sr
	}
	
    end if
    
    
    integer report_length = 0
    report_length = length(ddata)
    
    for x = 1 to report_length  do
	
	text_color(CYAN)
	printf(FN, "%s ", {ddata_labels[x]})
	
	if carpenters_frac then
	    
	    previous_unit = current_length_unit -- so we can convert back
	    new_unit = convert_unit(ddata[x], current_length_unit, "m")
	    ddata[x] = new_unit[3] -- now expressed in meters
	    
	    -- convert meters to carpenters inch fractions
	    text_color(YELLOW)
	    printf(FN,"%d %d/%d inches", carp(ddata[x], m_in, 32))
	    
	    -- convert back to "current_length_unit"
	    current_length_unit = previous_unit
	    new_unit = convert_unit(ddata[x], "m", current_length_unit)
	    ddata[x] = new_unit[3]
	    
	else
	    
	    text_color(YELLOW)
	    printf(FN, "%5.3f ", {ddata[x]})
	    text_color(BROWN)
	    printf(FN, "%s", {current_length_unit})
	    
	end if
	
    end for
    
    text_color(CYAN)    puts(FN, "\n\n\tFirst Floor Area:")
    if v = 3 then
	
	text_color(YELLOW)   printf(FN, " %5.2f ", dome_floor_area(dr,hxExtensions))
	text_color(BROWN)   printf(FN, "%s", {current_area_unit})
    
    elsif v = 2 then
	
	 text_color(YELLOW)   printf(FN, " %5.2f / %5.2f ", {(ATabc((dr-beam[2]), a, (dr-beam[2])))*10, PI*power(dr,2)} )
	 text_color(BROWN)   printf(FN, "%s", {current_area_unit})
    
    else
	 
	 text_color(YELLOW)   printf(FN, " %5.2f ", PI*power(.5*dd,2))
	 text_color(BROWN)   printf(FN, "%s", {current_area_unit})
    
    end if
    
    text_color(CYAN)    puts(FN, "\tCircumference:")
    text_color(YELLOW)   printf(FN, "\t%4.2f ", {PI*dd, current_length_unit})
    text_color(BROWN)   printf(FN, "%s", {current_length_unit})
    
    if v = 3 then
	
	text_color(CYAN)    puts(FN, "\n\t# of Hex wall extensions = ")
	text_color(YELLOW)  printf(FN, "%d", hxExtensions)
	text_color(CYAN)    puts(FN, " X ")
	text_color(YELLOW)  printf(FN,"%2.2f ", hxExArea())
	text_color(BROWN)   printf(FN, "%s", {current_area_unit})
    
    end if
    
    
    atom sphere_ratio
    if v = 3 or v = 32 then
	
	sphere_ratio = 4/9
	
    else
	
	sphere_ratio = .5
	
    end if
    
    text_color(CYAN)    puts(FN, "\n\n\tVolume:") -- uses 1/2 full sphere data
    text_color(YELLOW)   printf(FN, "\t%5.3f ", ((4/3*PI*power(sr,3))*sphere_ratio))
    text_color(BROWN)   printf(FN, "%s", {current_volume_unit})
    
    text_color(CYAN)    printf(FN, "\tSurface Area (%1.3f sphere):", sphere_ratio) -- uses sphere data
    text_color(YELLOW)    printf(FN, "\t%1.3f ",4*PI*power(sr,2)*sphere_ratio)
    text_color(BROWN)    printf(FN, "%s",{current_area_unit})
    
    text_color(CYAN) puts(FN, "\n\n\t# plywood sheets (based on surface area of sphere ratio) =")
    text_color(YELLOW) printf(FN, "\t%3.2f\n", ( ((4*PI*power(sr,2)*sphere_ratio + (PI*dd*rw)) / aps) ))
    
    if v = 3 then
	
	text_color(CYAN)
	puts(FN, "\n\t# of plywood sheets based on Area of Triangles + Riser Wall =")
	text_color(YELLOW)
	printf(FN, "\n\t%5.2f %s / %3.2f %s = %3.2f",{
	ceil((((lengths[1][1]*30) + (lengths[2][1]*45)) +(PI*dd*rw))),
	current_area_unit, aps, current_area_unit, 
	ceil(((lengths[1][1]*30) + (lengths[2][1]*45) + ceil(PI*dd*rw))) / aps})
	
	text_color(CYAN)
	puts(FN, "\n\n\tTimber = ")
	
	text_color(YELLOW)
	printf(FN, "%3.2f ", (30*a) + (30*lengths[1][2]) + (45*lengths[2][2]) + (55*b)+(80*c))
	
	text_color(BROWN)
	printf(FN, "linear %s\n\n", {current_length_unit})
	
    end if
end procedure

procedure domecalc()
    
    
    menu()
    set_lengths()
    report_dome_data()
    report_strut_lengths()
    
    
    text_color(CYAN)    puts(SCREEN, "\n\n\tCurrent Units of Measure: ")
    text_color(BROWN)   printf(SCREEN, "%s %s %s\n", {current_length_unit, 
							current_area_unit, 
							current_volume_unit
						    })
    
    
    text_color(GREEN)
    puts( FN, "\n\n\tThe following triangles compose the dome\'s \"skin\".\n\tIf Display of \"WOKEY\" plywood hub dome" &
    " is selected, \n\tthese edges will be slightly longer than the" &
    "\n\tmatching strut lengths.\n")
    
    sequence tlabels = {"Area", "HT", "A", "B", "A"}
    sequence qty = {30,45}
    
    if v = 2 then
	
	triangles = {"AAA", "BAB"}
	qty = {10,30}
	
    elsif v = 3 then
	
	tlabels = {"Area", "HT", "A", "B", "C"}
	triangles = {"ABA", "CBC", "wHvH"}
	qty = {30,45,10}
	
    elsif v = 312 then
	
	tlabels = {"Area", "HT", "A", "B", "A"}
	triangles = {"ABA", "CAC", "CBC"}
	qty = {30,30,15}
	
    elsif v = 4  then
	
	tlabels = {"Area", "HT", "A", "B", "A"}
	triangles = {"ABA", "CBC","CFD", "DFC", "DED", "EEE"}
	qty = { 30, 30, 30, 30, 30, 10 }    
	
    elsif v = 412 then
	
	-- 4v Class 1, Method 2:
	tlabels = {"Area", "HT", "A", "B", "A"}
	triangles = {"AEA", "DAC","CAD", "DED", "CBC", "BBB"}
	qty = { 30, 30, 30, 30, 30, 10 }    
	
	
    end if
    
    text_color(BROWN)
    for x = 1 to length(triangles) do
	
	text_color(YELLOW)
	printf(FN, "\n\t# %d %s:\t", {qty[x],triangles[x]})
	
	for i = 1 to 5 do
	    if v = 412 then
		if x = 1 then
		    tlabels = {"Area", "HT", "A", "E", "A"}
		elsif x = 2 then
		    tlabels = {"Area", "HT", "D", "A", "C"}
		elsif x = 3 then
		    tlabels = {"Area", "HT", "C", "A", "D"}
		elsif x = 4 then
		    tlabels = {"Area", "HT", "D", "E", "D"}
		elsif x = 5 then
		    tlabels = {"Area", "HT", "C", "B", "C"}
		elsif x = 6  then
		    tlabels = {"Area", "HT", "B", "B", "B"}
		end if
	    elsif v = 2 then
		if x = 1 then
		    tlabels = {"Area", "HT", "A", "A", "A"}
		elsif x = 2 then
		    tlabels = {"Area", "HT", "B", "A", "B"}
		end if
	    end if
	    
	    text_color(YELLOW)
	    printf(FN, "\t%s:", {tlabels[i]})
	    text_color(CYAN)
	    printf(FN, "%3.3f", {lengths[x][i]})
	    
	end for
	
    end for
	
	-- wHvH wedge under Hexagon vertex
	if v = 3 then
	    text_color(RED)
	    puts(FN, "\n\t(bevel wedge wHvH -6.79 degrees)")
	end if
    
    if HUB then
	hub_offset()
	report_strut_lengths()
	reset_lengths()
    end if
    
    if hardcopy then
	close(FN)
	hardcopy = FALSE
	FN = SCREEN
    end if
    
--      clear_screen()
    domecalc()
end procedure
----------------------------------------------------------------------
bk_color(BLACK)
clear_screen()
integer SCREEN = 1, hardcopy = FALSE, FN = SCREEN
sequence out

domecalc()
